using System.Security.Cryptography;

namespace LLM.HHData.Utils;

public static class UserAgentUtility
{
    public static string GetRandomUserAgent()
    {
        // Выбираем семейство
        var pick = NextInt(0, 6); // 0..5
        return pick switch
        {
            0 => ChromeWindows(),
            1 => ChromeMac(),
            2 => ChromeAndroid(),
            3 => SafariIOS(),
            4 => FirefoxDesktop(),
            _ => EdgeWindows()
        };
    }

    private static string ChromeWindows()
    {
        var chrome = ChromeVer();
        var platform = Pick(
        [
            "Windows NT 10.0; Win64; x64",
            "Windows NT 11.0; Win64; x64"
        ]);
        return $"Mozilla/5.0 ({platform}) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/{chrome} Safari/537.36";
    }

    private static string ChromeMac()
    {
        var chrome = ChromeVer();
        var mac = MacOsToken();
        return $"Mozilla/5.0 (Macintosh; {mac}) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/{chrome} Safari/537.36";
    }

    private static string ChromeAndroid()
    {
        var chrome = ChromeVer();
        var device = Pick(
        [
            "Pixel 7", "Pixel 7 Pro", "Pixel 8", "Pixel 8 Pro",
            "SM-S921B", "SM-S926B", "SM-S928B", // Galaxy S24 |+| Ultra
            "M2102J20SG" // Xiaomi placeholder
        ]);
        var androidVer = Pick(new[] { "13", "14" });
        return $"Mozilla/5.0 (Linux; Android {androidVer}; {device}) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/{chrome} Mobile Safari/537.36";
    }

    private static string SafariIOS()
    {
        var ios = Pick(new[] { "17_5", "17_6", "18_0", "18_1" });
        var mobileBuild = Pick(new[] { "15E148", "16E158", "19A346", "21E230" });
        var version = Pick(new[] { "17.5", "17.6", "18.0", "18.1" });
        return $"Mozilla/5.0 (iPhone; CPU iPhone OS {ios} like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/{version} Mobile/{mobileBuild} Safari/604.1";
    }

    private static string FirefoxDesktop()
    {
        var rv = NextInt(118, 131); 
        var platform = Pick(
        [
            "Windows NT 10.0; Win64; x64",
            "Windows NT 11.0; Win64; x64",
            "X11; Linux x86_64",
            "Macintosh; Intel Mac OS X " + MacOsVersionUnderscored()
        ]);
        return $"Mozilla/5.0 ({platform}; rv:{rv}.0) Gecko/20100101 Firefox/{rv}.0";
    }

    private static string EdgeWindows()
    {
        var edg = ChromeVer();  
        var platform = Pick(
        [
            "Windows NT 10.0; Win64; x64",
            "Windows NT 11.0; Win64; x64"
        ]);
        return $"Mozilla/5.0 ({platform}) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/{edg} Safari/537.36 Edg/{edg}";
    }

    // --- Helpers ---

    private static string ChromeVer()
    {
     
        var major = NextInt(120, 129);   
        var minor = NextInt(0, 1);      
        var build = NextInt(4000, 9000);
        var patch = NextInt(10, 199);
        return $"{major}.{minor}.{build}.{patch}";
    }

    private static string MacOsToken()
    {
       
        return "Intel Mac OS X " + MacOsVersionUnderscored();
    }

    private static string MacOsVersionUnderscored()
    {
        
        var major = Pick(new[] { 12, 13, 14, 15 });
        var minor = NextInt(0, major == 12 ? 7 : 7);  
        var patch = NextInt(0, 7); 
        return $"{major}_{minor}_{patch}";
    }

    private static int NextInt(int inclusiveMin, int inclusiveMax)
    {
        if (inclusiveMax < inclusiveMin) (inclusiveMin, inclusiveMax) = (inclusiveMax, inclusiveMin);
        return RandomNumberGenerator.GetInt32(inclusiveMin, inclusiveMax + 1);
    }

    private static T Pick<T>(IReadOnlyList<T> items)
    {
        if (items.Count == 0) throw new ArgumentException("Empty collection");
        var i = NextInt(0, items.Count - 1);
        return items[i];
    }
}