using System.Text;
using System.Text.Json;
using LLM.HHData.Config;
using LLM.HHData.Http;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Services;

public record VacancyListResponse(int found, int pages, int page, int per_page, VacancyItem[] items);
public record VacancyItem(string id);

public interface IVacancyListService
{
    Task<(List<string> ids, int found, int pages)> FetchAllAsync(string vacanciesUrl, CancellationToken ct);
}

public sealed class VacancyListService : IVacancyListService
{
    private readonly IHttpSender _http;
    private readonly AppConfig _cfg;
    private readonly ISystemLogService _log;
    private static readonly JsonSerializerOptions J = new() { PropertyNameCaseInsensitive = true };

    public VacancyListService(IOptions<AppConfig> cfg, IHttpSender http, ISystemLogService log)
    { _cfg = cfg.Value; _http = http; _log = log; }

    public async Task<(List<string> ids, int found, int pages)> FetchAllAsync(string vacanciesUrl, CancellationToken ct)
    {
        // гарантия наших параметров page/per_page поверх того, что пришло в URL
        string BuildPageUrl(int page)
        {
            var uri = new Uri(vacanciesUrl);
            var qb = new QueryBuilder(uri.Query);
            var perPageCfg = _cfg.Run.VacanciesPerPage;
            var perPage = perPageCfg > 0 ? Math.Min(perPageCfg, 100) : 100;

            qb.Set("page", page.ToString());
            qb.Set("per_page", perPage.ToString());

            // на случай отсутствия host/locale добавим их (не перезаписываем если уже есть)
            qb.TrySetIfMissing("host", _cfg.Run.Host);
            qb.TrySetIfMissing("locale", _cfg.Run.Locale);

            var baseUrl = uri.GetLeftPart(UriPartial.Path);
            return baseUrl + qb.ToString();
        }

        var allIds = new List<string>(capacity: 256);
        int page = 0, pages = 1, found = 0;

        while (page < pages && !ct.IsCancellationRequested)
        {
            var url = BuildPageUrl(page);
            var json = await _http.GetStringAsync(url, "application/json", ct);

            var list = JsonSerializer.Deserialize<VacancyListResponse>(json, J)
                       ?? new VacancyListResponse(0, 0, 0, 0, Array.Empty<VacancyItem>());

            if (page == 0)
            {
                found = list.found;
                pages = Math.Max(1, list.pages);
            }

            if (list.items is null || list.items.Length == 0)
                break; // пустая страница — завершаем

            allIds.AddRange(list.items.Select(i => i.id));

            page++;

            // лимит по количеству id для этого работодателя
            if (_cfg.Run.MaxVacancyItemsPerEmployer > 0 && allIds.Count >= _cfg.Run.MaxVacancyItemsPerEmployer)
            {
                allIds = allIds.Take(_cfg.Run.MaxVacancyItemsPerEmployer).ToList();
                break;
            }

            if (_cfg.Run.DelayBetweenVacancyPagesMs > 0 && page < pages)
                await Task.Delay(_cfg.Run.DelayBetweenVacancyPagesMs, ct);
        }

        return (allIds, found, pages);
    }


    // ─────────────────────────────────────────────────────────────────────────
    // helpers
    private sealed class QueryBuilder
    {
        private readonly Dictionary<string, string?> _map;

        public QueryBuilder(string query)
        {
            _map = new(StringComparer.OrdinalIgnoreCase);
            if (!string.IsNullOrEmpty(query))
            {
                var q = query.TrimStart('?').Split('&', StringSplitOptions.RemoveEmptyEntries);
                foreach (var kv in q)
                {
                    var i = kv.IndexOf('=');
                    if (i >= 0) _map[Uri.UnescapeDataString(kv[..i])] = Uri.UnescapeDataString(kv[(i + 1)..]);
                    else _map[Uri.UnescapeDataString(kv)] = null;
                }
            }
        }

        public void Set(string key, string value) => _map[key] = value;
        public void TrySetIfMissing(string key, string value)
        {
            if (!_map.ContainsKey(key)) _map[key] = value;
        }

        public override string ToString()
        {
            if (_map.Count == 0) return "";
            var sb = new StringBuilder("?");
            bool first = true;
            foreach (var (k, v) in _map)
            {
                if (!first) sb.Append('&'); first = false;
                sb.Append(Uri.EscapeDataString(k));
                if (v != null)
                {
                    sb.Append('=').Append(Uri.EscapeDataString(v));
                }
            }
            return sb.ToString();
        }
    }
}