using System.Net.Sockets;
using LLM.HHData.Config;
using LLM.HHData.Db;
using LLM.HHData.Http;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Services;

public interface IVacancyDetailService
{
    /// <summary>
    /// Получить полный JSON по одной вакансии.
    /// </summary>
    Task<string> GetDetailAsync(string vacancyId, CancellationToken ct);

    /// <summary>
    /// Последовательно обойти список id, вернуть успешно полученные id (для статистики).
    /// </summary>
    Task<int> FetchDetailsSequentialAsync(IEnumerable<string> vacancyIds, CancellationToken ct);
    Task<int> FetchDetailsConcurrentAsync(IReadOnlyList<string> ids, int degree, CancellationToken ct);
}

public sealed class VacancyDetailService : IVacancyDetailService
{
    private readonly AppConfig _cfg;
    private readonly IHttpSender _http;
    private readonly ISystemLogService _log;
    private readonly IKnownStateService _known;
    private readonly IRawStore _store;
    private readonly IRelatedVacanciesService _related;
    private readonly IEmployerEnqueuer _employerEnq;

    public VacancyDetailService(IOptions<AppConfig> cfg, IHttpSender http, ISystemLogService log, IKnownStateService knownState, IRawStore rawStore,
    IRelatedVacanciesService related,
    IEmployerEnqueuer employerEnqueuer)
    {
        _cfg = cfg.Value;
        _http = http;
        _log = log;
        _known = knownState;
        _store = rawStore;
        _related = related;
        _employerEnq = employerEnqueuer;
    }

    public Task<string> GetDetailAsync(string vacancyId, CancellationToken ct)
    {
        var url = BuildDetailUrl(vacancyId);
        return _http.GetStringAsync(url, "application/json", ct);
    }


    public async Task<int> FetchDetailsSequentialAsync(IEnumerable<string> vacancyIds, CancellationToken ct)
    {
        var ids = vacancyIds.ToList();
        int ok = 0, idx = 0;

        // если слишком много сетевых фейлов — эскалируем наверх,
        // чтобы верхний уровень мог «забанить» прокси и выдать новый
        int netErrors = 0;
        int netErrorEscalationThreshold = Math.Max(5, ids.Count / 3);

        var ttl = TimeSpan.FromHours(_cfg.Run.VacancyFreshTtlHours);

        foreach (var id in ids)
        {
            ct.ThrowIfCancellationRequested();
            idx++;

            if (!long.TryParse(id, out var vid))
            {
                await _log.WarnAsync($"[VAC-DETAIL {idx}/{ids.Count}] invalid vacancy id '{id}'");
                continue;
            }

            // свежая — только помечаем
            if (_known.IsVacancyFresh(vid, ttl))
            {
                await _store.TouchVacancyAsync(vid, ct);
                _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);
                continue;
            }

            if (_cfg.Run.DelayBetweenVacancyDetailsMs > 0 && idx > 1)
                await Task.Delay(_cfg.Run.DelayBetweenVacancyDetailsMs, ct);

            var url = BuildDetailUrl(id);

            // пер-вакансийные ретраи на транзиентные сетевые ошибки
            const int maxAttempts = 3;
            int attempts = 0;
            bool saved = false;

            while (attempts < maxAttempts && !saved)
            {
                attempts++;
                try
                {
                    ct.ThrowIfCancellationRequested();

                    var json = await _http.GetStringAsync(url, "application/json", ct);

                    var employerId = TryExtractEmployerId(json);
                    if (employerId.HasValue)
                    {
                        await _store.UpsertVacancyAsync(vid, employerId.Value, json, ct);
                        _known.MarkVacancySeen(vid);
                        _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);
                        ok++;
                        saved = true;

                        await FollowRelatedFromSeedAsync(vid, ct);
                    }
                    else
                    {
                        // нет employerId — считаем бизнес-ошибкой этой вакансии, но не сети
                        await _log.WarnAsync($"[VAC-DETAIL {idx}/{ids.Count}] employerId is null for vacancy {vid}");
                        saved = true; // чтобы не ретраить бесконечно
                    }
                }
                catch (OperationCanceledException oce) when (!ct.IsCancellationRequested && oce.Message.StartsWith("Proxy banned"))
                {
                    // важное: не глотаем «бан прокси» — пробрасываем наверх,
                    // чтобы сработала общая логика смены прокси
                    throw;
                }
                catch (OperationCanceledException) when (ct.IsCancellationRequested)
                {
                    throw; // корректно выходим по отмене
                }
                catch (Exception ex) when (ex is IOException || ex is HttpRequestException || ex is SocketException || ex is TaskCanceledException)
                {
                    // транзиентная сетевая ошибка
                    await _log.WarnAsync($"[VAC-DETAIL {idx}/{ids.Count}] {vid} attempt {attempts}/{maxAttempts} network error: {ex.Message}");

                    if (attempts >= maxAttempts)
                    {
                        netErrors++;
                        // не сохраняем; пойдём к следующей вакансии —
                        // но дадим шанс верхнему уровню сменить прокси, если ошибок много
                        break;
                    }

                    // экспоненциальная задержка
                    var backoffMs = 250 * attempts;
                    await Task.Delay(backoffMs, ct);
                }
                catch (Exception ex)
                {
                    // не-сетевые/бизнес-ошибки — логируем и переходим к следующей вакансии
                    await _log.WarnAsync($"[VAC-DETAIL {idx}/{ids.Count}] {vid} failed: {ex.Message}");
                    // не считаем это сетевым фейлом для эскалации
                    break;
                }
            }

            // лимит по количеству деталей на работодателя
            if (_cfg.Run.MaxVacancyDetailsPerEmployer > 0 && ok >= _cfg.Run.MaxVacancyDetailsPerEmployer)
                break;

            // если сетевых ошибок накопилось слишком много — эскалируем,
            // чтобы верхний уровень (Agent/WithProxyGuard) мог забанить прокси
            if (netErrors >= netErrorEscalationThreshold)
                throw new IOException($"[VAC-DETAIL] too many network failures: {netErrors}/{ids.Count}");
        }

        return ok;
    }

    private string BuildDetailUrl(string vacancyId)
            => $"{_cfg.BaseUrls.ApiVacancies?.TrimEnd('/')}/{Uri.EscapeDataString(vacancyId)}?host={_cfg.Run.Host}&locale={_cfg.Run.Locale}";

    private static long? TryExtractEmployerId(string json)
    {
        try
        {
            using var doc = System.Text.Json.JsonDocument.Parse(json);
            if (doc.RootElement.TryGetProperty("employer", out var emp) &&
                emp.TryGetProperty("id", out var idProp) &&
                idProp.ValueKind == System.Text.Json.JsonValueKind.String &&
                long.TryParse(idProp.GetString(), out var eid))
                return eid;
        }
        catch { }
        return null;
    }

    public async Task<int> FetchDetailsConcurrentAsync(IReadOnlyList<string> ids, int degree, CancellationToken ct)
    {
        if (ids == null || ids.Count == 0) return 0;

        // нормализуем степень параллелизма
        if (degree <= 0) degree = Environment.ProcessorCount;

        // учтём общий лимит деталей на работодателя
        int limit = _cfg.Run.MaxVacancyDetailsPerEmployer > 0
            ? Math.Min(_cfg.Run.MaxVacancyDetailsPerEmployer, ids.Count)
            : ids.Count;

        var list = ids.Take(limit).ToList();
        var ttl = TimeSpan.FromHours(_cfg.Run.VacancyFreshTtlHours);

        // если параллелизм = 1 — идём простой последовательной дорожкой (как в Sequential)
        if (degree == 1)
            return await FetchDetailsSequentialAsync(list, ct);

        var ok = 0;

        using var sem = new SemaphoreSlim(degree, degree);
        var tasks = new List<Task>(list.Count);

        // локальная задержка между СТАРТАМИ задач (дополнительно к пер-прокси троттлингу в HttpSender)
        var perItemDelay = _cfg.Run.DelayBetweenVacancyDetailsMs > 0
            ? TimeSpan.FromMilliseconds(_cfg.Run.DelayBetweenVacancyDetailsMs)
            : TimeSpan.Zero;
        DateTimeOffset nextStartAt = DateTimeOffset.MinValue;

        for (int i = 0; i < list.Count; i++)
        {
            ct.ThrowIfCancellationRequested();
            var id = list[i];

            // Расставим стартовые слоты, чтобы не стрелять пачкой
            if (perItemDelay > TimeSpan.Zero)
            {
                var now = DateTimeOffset.UtcNow;
                if (now < nextStartAt)
                    await Task.Delay(nextStartAt - now, ct);
                nextStartAt = (now > nextStartAt ? now : nextStartAt) + perItemDelay;
            }

            await sem.WaitAsync(ct);

            tasks.Add(Task.Run(async () =>
            {
                try
                {
                    ct.ThrowIfCancellationRequested();

                    if (!long.TryParse(id, out var vid))
                    {
                        await _log.WarnAsync($"Invalid vacancy id '{id}'");
                        return; // НЕ break — просто пропускаем
                    }

                    // СКИП свежей
                    if (_known.IsVacancyFresh(vid, ttl))
                    {
                        await _store.TouchVacancyAsync(vid, ct);
                        _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);
                        return; // НЕ break — просто пропускаем
                    }

                    // запрос JSON
                    var url = BuildDetailUrl(id);
                    var json = await _http.GetStringAsync(url, "application/json", ct);

                    // upsert
                    var employerId = TryExtractEmployerId(json) ?? 0;
                    await _store.UpsertVacancyAsync(vid, employerId, json, ct);
                    _known.MarkVacancySeen(vid);
                    _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);

                    Interlocked.Increment(ref ok);

                    await FollowRelatedFromSeedAsync(vid, ct);
                }
                catch (Exception ex)
                {
                    await _log.WarnAsync($"[VAC-DETAIL] {id} failed: {ex.Message}");
                }
                finally
                {
                    sem.Release();
                }
            }, ct));
        }

        await Task.WhenAll(tasks);
        await _log.InfoAsync($"[VAC-DETAIL] concurrent done: {ok}/{list.Count}");

        return ok;
    }

    private async Task<bool> SaveVacancyDetailNoFollowAsync(long vid, CancellationToken ct)
    {
        var ttl = TimeSpan.FromHours(_cfg.Run.VacancyFreshTtlHours);

        if (_known.IsVacancyFresh(vid, ttl))
        {
            await _store.TouchVacancyAsync(vid, ct);
            _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);
            return true;
        }

        var url = BuildDetailUrl(vid.ToString());
        var json = await _http.GetStringAsync(url, "application/json", ct);

        var employerId = TryExtractEmployerId(json) ?? 0;
        await _store.UpsertVacancyAsync(vid, employerId, json, ct);
        _known.MarkVacancySeen(vid);
        _known.UpdateVacancyLastSeen(vid, DateTimeOffset.UtcNow);
        return true;
    }

    private async Task FollowRelatedFromSeedAsync(long seedVacancyId, CancellationToken ct)
    {
        if (!_cfg.Run.FollowRelatedVacancies) return;

        var (relVacIds, relEmpIds, _, _) =
            await _related.FetchAllAsync(_http, seedVacancyId, _cfg.Run.Host, _cfg.Run.Locale, ct);


        var _newCount = 0;
        // 1) новые работодатели — в канал
        foreach (var eid in relEmpIds)
        {
            if (!_known.IsKnownEmployer(eid))
            {
                _known.MarkEmployerSeen(eid);
                await _employerEnq.EnqueueAsync(eid.ToString(), ct);
            }
        }

        if(_newCount >0)
            await _log.InfoAsync($"[RELATED] enqueue {_newCount} employers from seed vacancy {seedVacancyId}");


        // 2) новые вакансии — докачать ограниченно и БЕЗ дальнейшего follow
        if (_cfg.Run.MaxRelatedVacancyDetailsPerSeed <= 0) return;

        int taken = 0;
        foreach (var vid in relVacIds)
        {
            if (_known.IsKnownVacancy(vid))
                continue;

            try
            {
                await SaveVacancyDetailNoFollowAsync(vid, ct);
                taken++;
                if (taken >= _cfg.Run.MaxRelatedVacancyDetailsPerSeed)
                    break;
            }
            catch (Exception ex)
            {
                await _log.WarnAsync($"[RELATED] save detail {vid} failed: {ex.Message}");
            }
        }
    }

}