using System.Text.Json;
using LLM.HHData.Config;
using LLM.HHData.Http;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Services;

public interface IRelatedVacanciesService
{
    /// <summary>
    /// Возвращает (vacancyIds, employerIds, found, pages) для related-вакансий указанной вакансии.
    /// </summary>
    Task<(List<long> vacancyIds, HashSet<long> employerIds, int found, int pages)>
        FetchAllAsync(IHttpSender sender, long vacancyId, string host, string locale, CancellationToken ct);
}

public sealed class RelatedVacanciesService : IRelatedVacanciesService
{
    private readonly AppConfig _cfg;
    private readonly ISystemLogService _log;

    public RelatedVacanciesService(IOptions<AppConfig> cfg, ISystemLogService log)
    {
        _cfg = cfg.Value;
        _log = log;
    }

    public async Task<(List<long> vacancyIds, HashSet<long> employerIds, int found, int pages)>
        FetchAllAsync(IHttpSender sender, long vacancyId, string host, string locale, CancellationToken ct)
    {
        var allVacancyIds = new List<long>(256);
        var allEmployerIds = new HashSet<long>();

        var page = 0;
        var perPage = 100;
        var found = 0;
        var pages = 1;

        do
        {
            var url = $"{_cfg.BaseUrls.ApiVacancies.TrimEnd('/')}/{vacancyId}/related_vacancies?host={Uri.EscapeDataString(host)}&locale={Uri.EscapeDataString(locale)}&page={page}&per_page={perPage}";
            var json = await sender.GetStringAsync(url, "application/json", ct);

            using var doc = JsonDocument.Parse(json);
            if (page == 0)
            {
                found = doc.RootElement.TryGetProperty("found", out var f) && f.TryGetInt32(out var fv) ? fv : 0;
                pages = doc.RootElement.TryGetProperty("pages", out var p) && p.TryGetInt32(out var pv) ? pv : 1;
            }

            if (doc.RootElement.TryGetProperty("items", out var items) && items.ValueKind == JsonValueKind.Array)
            {
                foreach (var it in items.EnumerateArray())
                {
                    if (it.TryGetProperty("id", out var idProp) && idProp.ValueKind == JsonValueKind.String &&
                        long.TryParse(idProp.GetString(), out var vid))
                        allVacancyIds.Add(vid);

                    if (it.TryGetProperty("employer", out var emp) &&
                        emp.ValueKind == JsonValueKind.Object &&
                        emp.TryGetProperty("id", out var eidProp) &&
                        eidProp.ValueKind == JsonValueKind.String &&
                        long.TryParse(eidProp.GetString(), out var eid))
                        allEmployerIds.Add(eid);
                }
            }

            page++;
        } while (page < pages && page < 1000);

        await _log.InfoAsync($"related_vacancies[{vacancyId}]: found={found}, pages={pages}, collected={allVacancyIds.Count} ids, employers={allEmployerIds.Count}");
        return (allVacancyIds, allEmployerIds, found, pages);
    }
}
