using System;
using System.Collections.Concurrent;
using LLM.HHData.Db;

namespace LLM.HHData.Services;

public interface IKnownStateService
{
    Task InitializeAsync(CancellationToken ct);
    bool IsKnownEmployer(long employerId);
    bool IsKnownVacancy(long vacancyId);
    bool IsVacancyFresh(long vacancyId, TimeSpan ttl);
    void MarkEmployerSeen(long employerId);
    void MarkVacancySeen(long vacancyId);
    void UpdateVacancyLastSeen(long vacancyId, DateTimeOffset ts);
    IReadOnlyCollection<long> EmployersSnapshot();
    IReadOnlyCollection<long> VacanciesSnapshot();
}

public sealed class KnownStateService : IKnownStateService
{
    private readonly IRawStore _store;
    private readonly ISystemLogService _log;
    private readonly ConcurrentDictionary<long, byte> _employers = new();
    private readonly ConcurrentDictionary<long, byte> _vacancies = new();
    private readonly ConcurrentDictionary<long, DateTimeOffset> _vacancyLastSeen = new();


    public KnownStateService(IRawStore store, ISystemLogService log)
    {
        _store = store; _log = log;
    }

    public async Task InitializeAsync(CancellationToken ct)
    {
        var emp = await _store.LoadKnownEmployerIdsAsync(ct);
        foreach (var id in emp) _employers.TryAdd(id, 1);

        var vac = await _store.LoadKnownVacancyIdsAsync(ct);
        foreach (var id in vac) _vacancies.TryAdd(id, 1);

        var stamps = await _store.LoadVacancyLastSeenAsync(ct);
        foreach (var kv in stamps) _vacancyLastSeen[kv.Key] = kv.Value;

        await _log.InfoAsync($"Filled with {emp.Count} employes and {vac.Count} vacansies, { stamps.Count} are in lastSeen TTL");
    }

    public bool IsKnownEmployer(long employerId) => _employers.ContainsKey(employerId);
    public bool IsKnownVacancy(long vacancyId) => _vacancies.ContainsKey(vacancyId);

    public bool IsVacancyFresh(long vacancyId, TimeSpan ttl)
    {
        if (_vacancyLastSeen.TryGetValue(vacancyId, out var ts))
            return (DateTimeOffset.UtcNow - ts) < ttl;
        return false;
    }

    public void MarkEmployerSeen(long employerId) => _employers[employerId] = 1;
    public void MarkVacancySeen(long vacancyId) => _vacancies[vacancyId] = 1;

    public void UpdateVacancyLastSeen(long vacancyId, DateTimeOffset ts)
        => _vacancyLastSeen[vacancyId] = ts;

    public IReadOnlyCollection<long> EmployersSnapshot() => _employers.Keys.ToArray();
    public IReadOnlyCollection<long> VacanciesSnapshot() => _vacancies.Keys.ToArray();
}