using System.Text.Json;
using LLM.HHData.Config;
using LLM.HHData.Db;
using LLM.HHData.Http;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Services;

// Services/IEmployerService.cs
public interface IEmployerService
{

#if _
    Task<int> FetchProfilesSequentialAsync(IEnumerable<string> employerIds, CancellationToken ct);
#endif
    /// <summary>Скачать JSON одного работодателя, сохранить raw, извлечь и вернуть vacancies_url.</summary>
    Task<string?> FetchProfileAsync(string employerId, CancellationToken ct);
}
public sealed class EmployerService : IEmployerService
{
    private readonly AppConfig _cfg;
    private readonly IHttpSender _http;
    private readonly ISystemLogService _log;
    private readonly IEmployerVacancyIndex _index;
    private readonly IKnownStateService _known;
    private readonly IRawStore _store;

    public EmployerService(IOptions<AppConfig> cfg, IHttpSender http, ISystemLogService log, IEmployerVacancyIndex vacancyIndex, IKnownStateService known, IRawStore rawStore)
    {
        _cfg = cfg.Value;
        _http = http;
        _log = log;
        _index = vacancyIndex;
        _known = known;
        _store = rawStore;
    }

    public async Task<string?> FetchProfileAsync(string employerId, CancellationToken ct)
    {
        var url = $"{_cfg.BaseUrls.ApiEmployers}/{Uri.EscapeDataString(employerId)}?host={_cfg.Run.Host}&locale={_cfg.Run.Locale}";

       url += "&check_personal_data_resale=true&increment_views_counter=true&with_chat_info=true&with_complaint_about_employer=true&with_online_users_count=true&with_responses_count=true&with_skills_match=true";

       // await _log.InfoAsync($"[EMP] GET {url}");

        var json = await _http.GetStringAsync(url, "application/json", ct);

        // сохранить raw + отметки
        if (long.TryParse(employerId, out var eid))
        {
            await _store.UpsertEmployerAsync(eid, json, ct);
            _known.MarkEmployerSeen(eid);
        }

        // вытащить vacancies_url
        var vacanciesUrl = ExtractVacanciesUrl(json);
        if (!string.IsNullOrWhiteSpace(vacanciesUrl))
        {
            _index.Upsert(employerId, vacanciesUrl!);
        //   await  _log.InfoAsync($"[EMP] vacancies_url: {vacanciesUrl}");
        }
        else
        {
            await _log.WarnAsync($"[EMP] vacancies_url missing for employerId={employerId}");
        }
        return vacanciesUrl;
    }

#if _
    public async Task<int> FetchProfilesSequentialAsync(IEnumerable<string> employerIds, CancellationToken ct)
    {
        int ok = 0;
        int idx = 0;
        int withUrl = 0;
        var ids = employerIds.ToList();

        foreach (var id in ids)
        {
            if (_cfg.Run.DelayBetweenEmployersMs > 0)
                await Task.Delay(_cfg.Run.DelayBetweenEmployersMs, ct);

            ct.ThrowIfCancellationRequested();

            idx++;

            var url = $"{_cfg.BaseUrls.ApiEmployers}/{Uri.EscapeDataString(id)}?host={_cfg.Run.Host}&locale={_cfg.Run.Locale}";

           // await _log.InfoAsync($"[EMP {idx}/{ids.Count}] GET {url}");

            try
            {
                var json = await _http.GetStringAsync(url, "application/json", ct);

                if (long.TryParse(id, out var eid))
                {
                    await _store.UpsertEmployerAsync(eid, json, ct);
                    _known.MarkEmployerSeen(eid);
                }

                var vacanciesUrl = ExtractVacanciesUrl(json);

                if (!string.IsNullOrWhiteSpace(vacanciesUrl))
                {
                    _index.Upsert(id, vacanciesUrl!);
                    withUrl++;
                   // await _log.InfoAsync($"vacancies_url: {vacanciesUrl}");
                }
                else
                {
                    await _log.WarnAsync($"vacancies_url missing for employerId={id}");
                }

                // (при желании позже сохраним raw json в БД)

                ok++;
            }
            catch (Exception ex)
            {
                // не валим весь прогон; логируем и идём дальше
                await _log.ExceptionAsync(ex, new { msg = $"failed for employerId={id}" });
            }
        }

      //  await _log.InfoAsync($"Employer profiles fetched: {ok}/{ids.Count}");

        return ok;
    }

#endif

    private static string? ExtractVacanciesUrl(string json)
    {
        try
        {
            using var doc = JsonDocument.Parse(json);
            var root = doc.RootElement;

            if (root.TryGetProperty("vacancies_url", out var vu) && vu.ValueKind == JsonValueKind.String)
                return vu.GetString();

            // fallback: иногда может приходить внутри ссылок без locale/host — на всякий случай возвращаем null
            return null;
        }
        catch
        {
            return null;
        }
    }
}