using System.Threading.Channels;

namespace LLM.HHData.Services;

/// <summary>
/// Единый канал для employerId (string), общий на всё приложение:
/// IngestWorker в него пишет из sitemap,
/// EmployerAgent читает из него,
/// EmployerEnqueuer позволяет любому коду ДОкидывать employerId по ходу работы.
/// </summary>
public sealed class EmployersChannelHolder
{
    public Channel<string> Channel { get; }

    public EmployersChannelHolder()
    {
        Channel = System.Threading.Channels.Channel.CreateUnbounded<string>(
            new UnboundedChannelOptions
            {
                SingleReader = false,
                SingleWriter = false,
                AllowSynchronousContinuations = false
            });
    }
}

public interface IEmployerEnqueuer
{
    /// <summary>Положить employerId в общий канал.</summary>
    ValueTask EnqueueAsync(string employerId, CancellationToken ct = default);
}

public sealed class EmployerEnqueuer : IEmployerEnqueuer
{
    private readonly ChannelWriter<string> _writer;
    private readonly WorkTracker _tracker;

    public EmployerEnqueuer(EmployersChannelHolder holder, WorkTracker tracker)
    {
        _writer = holder.Channel.Writer;
        _tracker = tracker;
    }

    public ValueTask EnqueueAsync(string employerId, CancellationToken ct = default)
    {
        _tracker.Increment();               // ← добавили
        return _writer.WriteAsync(employerId, ct);
    }
}
