using System.Net.Sockets;
using System.Threading.Channels;
using LLM.HHData.Config;
using LLM.HHData.Db;
using LLM.HHData.Http;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Services;

public sealed class EmployerAgent
{
    private readonly ProxyInfo _proxy;
    private readonly IHttpSender _http;
    private readonly AppConfig _cfg;
    private readonly ISystemLogService _log;
    private readonly IEmployerVacancyIndex _index;
    private readonly IKnownStateService _known;
    private readonly IRawStore _store;

    private readonly IEmployerService _employers;
    private readonly IVacancyListService _vacList;
    private readonly IVacancyDetailService _vacDetail;

    private readonly IProxyLeaser _leaser;

    private readonly IEmployerEnqueuer _employerEnq;
    private readonly IRelatedVacanciesService _related;
    private readonly WorkTracker _work;


    public EmployerAgent(
        ProxyInfo proxy,
        IHttpSender http,
        IOptions<AppConfig> cfg,
        ISystemLogService log,
        IEmployerVacancyIndex index,
        IKnownStateService known,
        IRawStore store, IProxyLeaser leaser,
           IEmployerEnqueuer employerEnqueuer,
            IRelatedVacanciesService related, WorkTracker workTracker)
    {
        _proxy = proxy;
        _http = http;
        _cfg = cfg.Value;
        _log = log;
        _index = index;
        _known = known;
        _store = store;
        _leaser = leaser;
        _employerEnq = employerEnqueuer;
        _related = related;
        _work = workTracker;

        // создаём сервисы, «привязанные» к нашему IHttpSender
        _employers = new EmployerService(cfg, _http, _log, _index, _known, _store);
        _vacList = new VacancyListService(cfg, _http, _log);
        _vacDetail = new VacancyDetailService(cfg, _http, _log, _known, _store, _related, _employerEnq);

    }

    private async Task<T> WithProxyGuardAsync<T>(Func<Task<T>> action)
    {
        try
        {
            var res = await action();
            _leaser.RegisterSuccess(_proxy);
            return res;
        }
        catch (HttpRequestException hre) when (hre.InnerException is IOException || hre.InnerException is SocketException)
        {
            if (_leaser.RegisterFailure(_proxy))
                throw new OperationCanceledException("Proxy banned due to repeated failures", hre);
            throw;
        }
        catch (IOException ioex)
        {
            if (_leaser.RegisterFailure(_proxy))
                throw new OperationCanceledException("Proxy banned due to repeated failures", ioex);
            throw;
        }
    }

    public async Task RunAsync(ChannelReader<string> employers, CancellationToken ct)
    {
        await _log.InfoAsync($"[AGENT # {_proxy.Host}:{_proxy.Port}] started");

        while (await employers.WaitToReadAsync(ct))
        {
            while (employers.TryRead(out var empId))
            {
                try
                {
                    ct.ThrowIfCancellationRequested();

                    // 1) профиль работодателя
                    var vurl = await WithProxyGuardAsync(() => _employers.FetchProfileAsync(empId, ct));

                    if (_cfg.Run.DelayBetweenEmployersMs > 0)
                        await Task.Delay(_cfg.Run.DelayBetweenEmployersMs, ct);

                    if (string.IsNullOrWhiteSpace(vurl))
                    {
                        await _log.WarnAsync($"[AGENT {_proxy.Host}] employer {empId}: vacancies_url missing");
                        continue;
                    }

                    // 2) id вакансий по пагинации
                    var (ids, found, pages) = await WithProxyGuardAsync(() => _vacList.FetchAllAsync(vurl, ct));

                    //                    await _log.InfoAsync($"[AGENT {_proxy.Host}] EMP {empId} ids={ids.Count}/{found} pages={pages}");

                    if (found > 0 && ids.Count > 0)
                        await WithProxyGuardAsync(async () =>
                        {
                            var deg = Math.Max(1, _cfg.Run.MaxDegreeVacancyDetailsPerEmployer);
                            await _vacDetail.FetchDetailsConcurrentAsync(ids, deg, ct);
                            return 0;
                        });
                }
                catch (Exception ex)
                {
                    await _log.WarnAsync($"[AGENT {_proxy.Host}] EMP {empId} failed: {ex.Message}");
                    // тут не баним прокси автоматически — решение принимает верхний уровень
                }
                finally
                {
                    _work.DecrementAndMaybeComplete();     // ← декремент «висящей» работы
                }
            }
        }

        await _log.InfoAsync($"[AGENT {_proxy.Host}:{_proxy.Port}] stopped");
    }
}