﻿using LLM.HHData.Config;
using LLM.HHData.Db;
using LLM.HHData.Http;
using LLM.HHData.Services;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;

namespace LLM.HHData;

internal class Program
{
    private static async Task Main(string[] args)
    {
        var builder = Host.CreateDefaultBuilder(args)
    .ConfigureAppConfiguration(cfg =>
    {
        cfg.Sources.Clear();
        cfg.AddJsonFile("config.json", optional: false, reloadOnChange: true);
        cfg.AddEnvironmentVariables("APP_");
        if (args.Length > 0) cfg.AddCommandLine(args);
    })
    .ConfigureLogging(l =>
    {
        l.ClearProviders();
        l.AddSimpleConsole(o =>
        {
            o.SingleLine = true;
            o.TimestampFormat = "HH:mm:ss ";
        });
        l.SetMinimumLevel(LogLevel.Warning);
    })
    .ConfigureServices((ctx, services) =>
    {
        services.Configure<AppConfig>(ctx.Configuration);

        services.AddSingleton<ISystemLogService, SystemLogService>();
        services.AddSingleton<IHttpSender, HttpSender>();
        
        services.AddSingleton<IProxyProvider>(sp =>
            new FileProxyProvider(sp.GetRequiredService<IOptions<AppConfig>>().Value.Http.ProxyFilePath));

        services.AddSingleton<ISitemapService, SitemapService>();

        services.AddSingleton<IEmployerVacancyIndex, InMemoryEmployerVacancyIndex>();
        services.AddSingleton<IEmployerService, EmployerService>();

        services.AddSingleton<IVacancyListService, VacancyListService>();
        services.AddSingleton<IVacancyDetailService, VacancyDetailService>();

        services.AddSingleton<IDbFactory, DbFactory>(sp =>
        {
            var cfg = sp.GetRequiredService<IOptions<AppConfig>>().Value;
            return new DbFactory(cfg);
        });


        services.AddSingleton<IKnownStateService, KnownStateService>();

        services.AddSingleton<IRelatedVacanciesService, RelatedVacanciesService>();
        services.AddSingleton<EmployersChannelHolder>();
        // ➕ синглтон-паблишер (обёртка над Channel.Writer)
        services.AddSingleton<IEmployerEnqueuer, EmployerEnqueuer>();
        services.AddSingleton<WorkTracker>();


        services.AddSingleton<IBatchedRawStore, BatchedRawStore>();
        services.AddSingleton<IRawStore>(sp => new BatchedRawStoreAdapter(
            sp.GetRequiredService<IBatchedRawStore>(), sp.GetRequiredService<IDbFactory>(), sp.GetRequiredService<ISystemLogService>()));

        services.AddSingleton<IProxyLeaser, ProxyLeaser>();
        services.AddSingleton<IHttpSenderFactory, HttpSenderFactory>();

        services.AddHostedService<IngestWorker>();
    });

        await builder.Build().RunAsync();
    }
}