using LLM.HHData.Config;
using LLM.HHData.Db;
using LLM.HHData.Http;
using LLM.HHData.Services;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Options;

namespace LLM.HHData;

public sealed class IngestWorker : BackgroundService
{
    private readonly ISystemLogService _log;
    private readonly IHttpSender _http;
    private readonly ISitemapService _sitemaps;
    private readonly IEmployerService _employers;
    private readonly IEmployerVacancyIndex _index;
    private readonly IVacancyListService _vacList;
    private readonly IVacancyDetailService _vacDetail;
    private readonly IKnownStateService _known;
    private readonly AppConfig _cfg;
    private readonly IRawStore _rawStore;
    private readonly IHttpSenderFactory _httpSenderFactory;
    private readonly IProxyLeaser _leaser;
    private readonly EmployersChannelHolder _employersHolder;
    private readonly IEmployerEnqueuer _employerEnq;
    private readonly IRelatedVacanciesService _related;
    private readonly WorkTracker _work;

    public IngestWorker(
        ISystemLogService log,
        IHttpSender http,
        ISitemapService sitemaps,
        IEmployerService employers,
        IOptions<AppConfig> cfg,
        IEmployerVacancyIndex vacancyIndex,
        IVacancyListService vacancyListService,
        IVacancyDetailService vacancyDetail, IKnownStateService known,
        IRawStore rawStore, IHttpSenderFactory httpSenderFactory, IProxyLeaser proxyLeaser, EmployersChannelHolder employersChannel,
    IEmployerEnqueuer employerEnqueuer,
    IRelatedVacanciesService relatedVacancies, WorkTracker tracker)
    {
        _log = log;
        _http = http;
        _sitemaps = sitemaps;
        _employers = employers;
        _index = vacancyIndex;
        _vacList = vacancyListService;
        _vacDetail = vacancyDetail;
        _known = known;

        _rawStore = rawStore;
        _httpSenderFactory = httpSenderFactory;
        _leaser = proxyLeaser;

        _employersHolder = employersChannel;


        _employerEnq = employerEnqueuer;
        _related = relatedVacancies;

        _work = tracker;

        _cfg = cfg.Value;
    }

    // ─────────────────────────────────────────────────────────────────────────
    // ENTRY
    // ─────────────────────────────────────────────────────────────────────────
    protected override async Task ExecuteAsync(CancellationToken stoppingToken)
    {
        Guid iter = Guid.NewGuid();

        try
        {
            await _log.InfoAsync($"Iteration {iter} started");

            await _known.InitializeAsync(stoppingToken);
            await _log.InfoAsync($"Known employers: {_known.EmployersSnapshot().Count}, known vacancies: {_known.VacanciesSnapshot().Count}");

            await LogCookiesInfoAsync(stoppingToken);

            var xmlMain = await FetchRootSitemapAsync(stoppingToken);
            var sitemapUrls = _sitemaps.ExtractEmployerSitemapUrls(xmlMain).ToList();
            await _log.InfoAsync($"Employer sitemaps found: {sitemapUrls.Count}");

            var employerIds = await CollectEmployerIdsSequentialAsync(sitemapUrls, stoppingToken);

            employerIds = [.. employerIds.OrderBy(_ => Guid.NewGuid())];

            await _log.InfoAsync($"All XML done. Unique employers: {employerIds.Count}");

            await _log.InfoAsync($"Using {_leaser.Total} proxies to create agents");

            if (_cfg.Run.UseProxyAgents)
            {
                await ProcessEmployersPipelineAsync(employerIds, stoppingToken);
            }
            else
            {
                // твоя старая последовательная версия (оставляем как есть)
                await ProcessEmployersPipelineAsync_Sequensal(employerIds, stoppingToken);
            }


            _index.Flush();

            await _log.InfoAsync($"Iteration {iter} done");

        }
        catch (Exception ex)
        {
            await _log.ExceptionAsync(ex);
            await _log.InfoAsync($"Iteration {iter} failed");
            throw;
        }
    }

    private async Task ProcessEmployersPipelineAsync(HashSet<string> employerIds, CancellationToken ct)
    {
        // БЕРЕМ общий канал из holder
        var chan = _employersHolder.Channel;

        // Producer
        var producer = Task.Run(async () =>
        {
            foreach (var emp in employerIds)
            {
                ct.ThrowIfCancellationRequested();
                _work.Increment();                          // ← добавили
                await chan.Writer.WriteAsync(emp, ct);
            }
            _work.MarkSeedingCompleted();
        }, ct);

        // Agents
        var agents = new List<Task>();
        var want = Math.Min(_cfg.Run.MaxDegreeEmployers, _leaser.Total);

        for (int i = 0; i < want; i++)
        {
            ProxyInfo proxy;
            var spawnBackoff = TimeSpan.FromSeconds(1);
            while (!_leaser.TryLease(out proxy))
            {
                await _log.WarnAsync("[AGENT-SPAWN] no free proxies; waiting…");
                await Task.Delay(spawnBackoff, ct);
                if (spawnBackoff < TimeSpan.FromSeconds(10))
                    spawnBackoff += TimeSpan.FromSeconds(1);
            }

            agents.Add(Task.Run(async () =>
            {
                var reader = chan.Reader;

                while (!ct.IsCancellationRequested && !reader.Completion.IsCompleted)
                {
                    var http = _httpSenderFactory.CreateForProxy(proxy);

                    var agent = new EmployerAgent(
                        proxy, http, Options.Create(_cfg), _log, _index, _known, _rawStore, _leaser,
                        _employerEnq, _related, _work);

                    try
                    {
                        await agent.RunAsync(reader, ct);
                        _leaser.Return(proxy);
                        break;
                    }
                    catch (OperationCanceledException oce) when (oce.Message.StartsWith("Proxy banned"))
                    {
                        await _log.WarnAsync($"[AGENT {proxy.Host}] banned; switching proxy…");
                        ProxyInfo newProxy;
                        var wait = TimeSpan.FromSeconds(2);
                        while (!_leaser.TryLease(out newProxy))
                        {
                            await _log.WarnAsync("[AGENT] no free proxies; waiting…");
                            await Task.Delay(wait, ct);
                            if (wait < TimeSpan.FromSeconds(15)) wait += TimeSpan.FromSeconds(1);
                        }
                        proxy = newProxy;
                        continue;
                    }
                    catch (Exception ex)
                    {
                        await _log.WarnAsync($"[AGENT {proxy.Host}] agent error: {ex.Message} — will continue");
                        continue;
                    }
                }
            }, ct));
        }

        await producer;
        await Task.WhenAll(agents);
    }


    [Obsolete("No proxy option, must be removed")]
    private async Task ProcessEmployersPipelineAsync_Sequensal(HashSet<string> employerIds, CancellationToken ct)
    {
        int idx = 0;
        foreach (var empId in employerIds)
        {
            ct.ThrowIfCancellationRequested();
            idx++;
            await _log.InfoAsync($"[EMP {idx}/{employerIds.Count}] employerId={empId}");

            // 1) профиль работодателя (с сохранением raw)
            var vacanciesUrl = await _employers.FetchProfileAsync(empId, ct);

            // задержка между работодателями (если настроена)
            if (_cfg.Run.DelayBetweenEmployersMs > 0)
                await Task.Delay(_cfg.Run.DelayBetweenEmployersMs, ct);

            if (string.IsNullOrWhiteSpace(vacanciesUrl))
            {
                await _log.WarnAsync($"[EMP {empId}] vacancies_url missing — skip vacancies");
                continue;
            }

            // 2) листинг вакансий
            var (ids, found, pages) = await _vacList.FetchAllAsync(vacanciesUrl!, ct);
            await _log.InfoAsync($"[EMP {empId}] list: ids={ids.Count}/{found} pages={pages}");

            // 3) детали вакансий (со скипом свежих внутри сервиса)
            if (ids.Count > 0 && found > 0)
            {
                await _vacDetail.FetchDetailsSequentialAsync(ids, ct);
            }
        }
    }

    // ─────────────────────────────────────────────────────────────────────────
    // STAGE 0: cookies
    // ─────────────────────────────────────────────────────────────────────────
    private async Task LogCookiesInfoAsync(CancellationToken ct)
    {
        if (_cfg.Cookies.Enabled)
        {
            var count = (_http as HttpSender)?.PreviewCookieFileCount() ?? 0;
            await _log.InfoAsync($"Cookies enabled, file: {_cfg.Cookies.FilePath}, lines parsed: {count}");
        }
        else
        {
            await _log.WarnAsync("Cookies are disabled in config.");
        }
    }

    // ─────────────────────────────────────────────────────────────────────────
    // STAGE 1: root sitemap
    // ─────────────────────────────────────────────────────────────────────────
    private async Task<string> FetchRootSitemapAsync(CancellationToken ct)
    {
        var xmlMain = await _sitemaps.GetMainSitemapAsync(ct);
        await _log.InfoAsync($"Root sitemap ok, length={xmlMain.Length}");
        return xmlMain;
    }

    // ─────────────────────────────────────────────────────────────────────────
    // STAGE 2: employer ids из employer-*.xml (последовательно с паузой)
    // ─────────────────────────────────────────────────────────────────────────
    private async Task<HashSet<string>> CollectEmployerIdsSequentialAsync(
        List<string> sitemapUrls, CancellationToken ct)
    {
        var max = _cfg.Run.MaxEmployerSitemaps > 0 ? _cfg.Run.MaxEmployerSitemaps : sitemapUrls.Count;
        var delay = TimeSpan.FromSeconds(_cfg.Run.DelayBetweenXmlSeconds <= 0 ? 30 : _cfg.Run.DelayBetweenXmlSeconds);

        var allEmployerIds = new HashSet<string>();

        for (int i = 0; i < max; i++)
        {
            ct.ThrowIfCancellationRequested();

            var url = sitemapUrls[i];
            await _log.InfoAsync($"[{i + 1}/{max}] Fetch {url}");

            var ids = await _sitemaps.GetEmployerIdsFromSitemapAsync(url, ct);
            var before = allEmployerIds.Count;
            foreach (var id in ids) allEmployerIds.Add(id);
            var added = allEmployerIds.Count - before;

            await _log.InfoAsync($"  + {added} new, total employers: {allEmployerIds.Count}");

            if (i < max - 1)
            {
                // оставляю твой текущий «check»-блок как есть
                // if (added > 0)
                //      break;
                //  else
                //  {
                await _log.InfoAsync($"  sleep {delay.TotalSeconds}s before next XML…");
                await Task.Delay(delay, ct);
                //   }
            }
        }
        return allEmployerIds;
    }
    public override async Task StopAsync(CancellationToken cancellationToken)
    {
        // закрываем общий канал, чтобы агенты корректно вышли из циклов чтения
        _employersHolder.Channel.Writer.TryComplete();
        await base.StopAsync(cancellationToken);
    }
}
