using System.Net.Sockets;
using System.Text;

namespace LLM.HHData.Http;

/// <summary>
/// Минимальный SOCKS5-коннектор  
/// </summary>
public static class Socks5Connector
{
    public static async Task<Stream> ConnectAsync(
        string proxyHost, int proxyPort,
        string destHost, int destPort,
        string? user, string? pass,
        CancellationToken ct)
    {
        user = (user ?? string.Empty).Trim();
        pass = (pass ?? string.Empty).Trim();
        var haveCreds = user.Length > 0 || pass.Length > 0;

        var tcp = new TcpClient();
        using var ctr = ct.Register(() => { try { tcp.Dispose(); } catch { /* ignore */ } });

        await tcp.ConnectAsync(proxyHost, proxyPort, ct).ConfigureAwait(false);
        var stream = tcp.GetStream();

        // 1) greeting: list methods
        if (!haveCreds)
        {
            // Только "no auth"
            await stream.WriteAsync(new byte[] { 0x05, 0x01, 0x00 }, ct).ConfigureAwait(false);
        }
        else
        {
            // Предлагаем "no auth" и "user/pass"
            await stream.WriteAsync(new byte[] { 0x05, 0x02, 0x00, 0x02 }, ct).ConfigureAwait(false);
        }

        // 2) server method selection
        var ver = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        var method = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        if (ver != 0x05) throw new IOException($"SOCKS version mismatch: {ver}");
        if (method == 0xFF) throw new IOException("SOCKS5: no acceptable auth methods");

        if (method == 0x02) // RFC1929
        {
            if (!haveCreds)
                throw new IOException("SOCKS5 server requires auth but no credentials provided");

            // username/password MUST be <= 255 bytes (ASCII)
            var u = Encoding.ASCII.GetBytes(user);
            var p = Encoding.ASCII.GetBytes(pass);
            if (u.Length > 255 || p.Length > 255)
                throw new IOException("SOCKS5 auth: username/password length > 255");

            var auth = new byte[3 + u.Length + p.Length];
            auth[0] = 0x01; // subnegotiation version
            auth[1] = (byte)u.Length;
            Buffer.BlockCopy(u, 0, auth, 2, u.Length);
            var o = 2 + u.Length;
            auth[o++] = (byte)p.Length;
            Buffer.BlockCopy(p, 0, auth, o, p.Length);

            await stream.WriteAsync(auth, ct).ConfigureAwait(false);

            var av = await ReadByteAsync(stream, ct).ConfigureAwait(false); // should be 0x01
            var status = await ReadByteAsync(stream, ct).ConfigureAwait(false);
            if (av != 0x01 || status != 0x00)
                throw new IOException("SOCKS5 auth failed");
        }
        else if (method != 0x00)
        {
            // сервер выбрал что-то странное (например GSSAPI): не поддерживаем
            throw new IOException($"SOCKS5: unsupported auth method {method}");
        }

        // 3) CONNECT request
        var req = BuildConnectRequest(destHost, destPort);
        await stream.WriteAsync(req, ct).ConfigureAwait(false);

        // 4) reply
        var rVer = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        var rep = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        var rsv = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        var atyp = await ReadByteAsync(stream, ct).ConfigureAwait(false);
        if (rVer != 0x05 || rsv != 0x00)
            throw new IOException("SOCKS5 reply malformed");

        // читаем BND.ADDR/BND.PORT, но они нам не нужны — просто потребляем
        int addrLen = atyp switch
        {
            0x01 => 4,                   // IPv4
            0x03 => await ReadByteAsync(stream, ct).ConfigureAwait(false), // DOMAIN (len)
            0x04 => 16,                  // IPv6
            _ => throw new IOException($"SOCKS5 reply: unknown ATYP {atyp}")
        };
        if (atyp == 0x01 || atyp == 0x04)
        {
            await ReadExactAsync(stream, addrLen, ct).ConfigureAwait(false);
        }
        else
        {
            await ReadExactAsync(stream, addrLen, ct).ConfigureAwait(false);
        }
        await ReadExactAsync(stream, 2, ct).ConfigureAwait(false); // port

        if (rep != 0x00)
            throw new IOException($"SOCKS5 connect failed: {MapRep(rep)}");

        // успех — возвращаем поток; TcpClient отдаём на жизнь вместе с потоком
        return stream;
    }

    private static byte[] BuildConnectRequest(string destHost, int destPort)
    {
        // Попробуем распознать IPv4/IPv6. Иначе отправим как домен.
        if (System.Net.IPAddress.TryParse(destHost, out var ip))
        {
            if (ip.AddressFamily == AddressFamily.InterNetwork)
            {
                var addr = ip.GetAddressBytes();
                var req = new byte[4 + 4 + 2];
                req[0] = 0x05; req[1] = 0x01; req[2] = 0x00; req[3] = 0x01; // ATYP=IPv4
                Buffer.BlockCopy(addr, 0, req, 4, 4);
                req[^2] = (byte)((destPort >> 8) & 0xFF);
                req[^1] = (byte)(destPort & 0xFF);
                return req;
            }
            else
            {
                var addr = ip.GetAddressBytes();
                var req = new byte[4 + 16 + 2];
                req[0] = 0x05; req[1] = 0x01; req[2] = 0x00; req[3] = 0x04; // ATYP=IPv6
                Buffer.BlockCopy(addr, 0, req, 4, 16);
                req[^2] = (byte)((destPort >> 8) & 0xFF);
                req[^1] = (byte)(destPort & 0xFF);
                return req;
            }
        }

        // DOMAIN (ASCII)
        var hostBytes = Encoding.ASCII.GetBytes(destHost);
        if (hostBytes.Length > 255) throw new ArgumentException("Hostname too long for SOCKS5.");
        var reqDom = new byte[4 + 1 + hostBytes.Length + 2];
        reqDom[0] = 0x05; reqDom[1] = 0x01; reqDom[2] = 0x00; reqDom[3] = 0x03;
        reqDom[4] = (byte)hostBytes.Length;
        Buffer.BlockCopy(hostBytes, 0, reqDom, 5, hostBytes.Length);
        reqDom[^2] = (byte)((destPort >> 8) & 0xFF);
        reqDom[^1] = (byte)(destPort & 0xFF);
        return reqDom;
    }

    private static string MapRep(int rep) => rep switch
    {
        0x01 => "general failure",
        0x02 => "connection not allowed by ruleset",
        0x03 => "network unreachable",
        0x04 => "host unreachable",
        0x05 => "connection refused by destination host",
        0x06 => "TTL expired",
        0x07 => "command not supported / protocol error",
        0x08 => "address type not supported",
        _ => $"unknown error 0x{rep:X2}"
    };

    private static async Task<byte[]> ReadExactAsync(Stream s, int len, CancellationToken ct)
    {
        var buf = new byte[len];
        var read = 0;
        while (read < len)
        {
            var r = await s.ReadAsync(buf.AsMemory(read, len - read), ct).ConfigureAwait(false);
            if (r <= 0) throw new EndOfStreamException();
            read += r;
        }
        return buf;
    }

    private static async Task<int> ReadByteAsync(Stream s, CancellationToken ct)
    {
        var b = new byte[1];
        var r = await s.ReadAsync(b.AsMemory(0, 1), ct).ConfigureAwait(false);
        if (r != 1) throw new EndOfStreamException();
        return b[0];
    }
}
