namespace LLM.HHData.Http;

public record ProxyInfo(string Host, int Port, string Username, string Password);

public interface IProxyProvider
{
    ProxyInfo? GetRandomProxy();
    IReadOnlyList<ProxyInfo> GetAll();
    int Count { get; }
}

public sealed class FileProxyProvider : IProxyProvider
{
    private readonly List<ProxyInfo> _proxies;
    private readonly Random _rnd = new();

    public IReadOnlyList<ProxyInfo> GetAll() => _proxies;

    public int Count => _proxies == null ? 0 : _proxies.Count;

    public FileProxyProvider(string path)
    {
        if (!File.Exists(path))
        {
            _proxies = [];
            return;
        }

        _proxies = File.ReadAllLines(path)
            .Select(l => l.Trim())
            .Where(l => !string.IsNullOrEmpty(l) && !l.StartsWith("#"))
            .Select(ParseLine)
            .Where(p => p != null)
            .Cast<ProxyInfo>()
            .ToList();
    }

    private static ProxyInfo? ParseLine4(string line)
    {
        var parts = line.Split(':');
        if (parts.Length == 4 && int.TryParse(parts[1], out var port))
            return new ProxyInfo(parts[0], port, parts[2], parts[3]);
        return null;
    }

    private static ProxyInfo? ParseLine(string line)
    {
        if (string.IsNullOrWhiteSpace(line)) return null;
        var trimmed = line.Trim();
        if (trimmed.StartsWith("#")) return null; // комментарии

        var parts = trimmed.Split(':');
        // host:port:user:pass
        if (parts.Length == 4 && int.TryParse(parts[1], out var port4))
            return new ProxyInfo(parts[0], port4, parts[2], parts[3]);

        // host:port
        if (parts.Length == 2 && int.TryParse(parts[1], out var port2))
            return new ProxyInfo(parts[0], port2, Username: string.Empty, Password: string.Empty);

        // host:port:user  (редкий случай — пустой пароль)
        if (parts.Length == 3 && int.TryParse(parts[1], out var port3))
            return new ProxyInfo(parts[0], port3, parts[2], Password: string.Empty);

        return null;
    }

    public ProxyInfo? GetRandomProxy()
    {
        if (_proxies.Count == 0) return null;
        var idx = _rnd.Next(_proxies.Count);
        return _proxies[idx];
    }
}