using System.Collections.Concurrent;
using LLM.HHData.Config;
using LLM.HHData.Services;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Http;

public interface IProxyLeaser
{
    bool TryLease(out ProxyInfo proxy);
    void Return(ProxyInfo proxy, bool ban = false);
    int Total { get; }
    int Free { get; }
    bool RegisterFailure(ProxyInfo proxy); // true => забанен сейчас
    void RegisterSuccess(ProxyInfo proxy);
}

public sealed class ProxyLeaser : IProxyLeaser
{
    private readonly ConcurrentQueue<ProxyInfo> _free = new();
    private readonly ConcurrentDictionary<string, DateTimeOffset> _banned = new();
    private readonly ConcurrentDictionary<string, (int Count, DateTimeOffset First)> _strikes = new();

    private readonly TimeSpan _banTtl;
    private readonly TimeSpan _strikeWindow;
    private readonly int _strikeThreshold;
    private readonly int _total;
    private readonly ISystemLogService _log;

    public ProxyLeaser(IProxyProvider provider, IOptions<AppConfig> cfg, ISystemLogService log)
    {
        foreach (var p in provider.GetAll()) _free.Enqueue(p);
        _total = _free.Count;

        _banTtl = TimeSpan.FromMinutes(cfg.Value.Run.ProxyHealthCheckMinutes);
        _strikeWindow = TimeSpan.FromSeconds(cfg.Value.Run.ProxyStrikeWindowSec);
        _strikeThreshold = Math.Max(1, cfg.Value.Run.ProxyStrikeThreshold);

        _log = log;
    }

    public int Total => _total;

    public int Free { get { Cleanup(); return _free.Count; } }

    public bool TryLease(out ProxyInfo proxy)
    {
        Cleanup();

        int tries = _total + 3;

        while (tries-- > 0 && _free.TryDequeue(out proxy!))
        {
            if (!IsBanned(proxy)) return true;
        }
        proxy = default!;
        return false;
    }

    public void Return(ProxyInfo proxy, bool ban = false)
    {
        var key = Key(proxy);
        if (ban)
        {
            _banned[key] = DateTimeOffset.UtcNow + _banTtl;
            _strikes.TryRemove(key, out _);
            return;
        }
        _free.Enqueue(proxy);
    }

    // 👇 вызывать при сетевом фейле
    public bool RegisterFailure(ProxyInfo proxy)
    {
        var key = Key(proxy);
        var now = DateTimeOffset.UtcNow;

        _strikes.AddOrUpdate(key,
            _ => (1, now),
            (_, cur) =>
            {
                if (now - cur.First > _strikeWindow) return (1, now);
                return (cur.Count + 1, cur.First);
            });

        var s = _strikes[key];
        var msg = $"[PROXY] {proxy.Host}:{proxy.Port} fail {s.Count}/{_strikeThreshold}";
        _ = _log.WarnAsync(msg);

        if (s.Count >= _strikeThreshold)
        {
            _banned[key] = now + _banTtl;
            _strikes.TryRemove(key, out _);
            _ = _log.WarnAsync($"[PROXY] {proxy.Host}:{proxy.Port} banned for {_banTtl.TotalMinutes}m");
            return true;
        }

        return false;
    }

    // 👇 вызывать на успешном запросе
    public void RegisterSuccess(ProxyInfo proxy)
    {
        if (_strikes.TryRemove(Key(proxy), out _))
        {
            _ = _log.InfoAsync($"[PROXY] {proxy.Host}:{proxy.Port} recovered (fail counter reset)");
        }
    }

    private bool IsBanned(ProxyInfo p)
        => _banned.TryGetValue(Key(p), out var until) && until > DateTimeOffset.UtcNow;

    private void Cleanup()
    {
        var now = DateTimeOffset.UtcNow;
        foreach (var kv in _banned.ToArray())
            if (kv.Value <= now) _banned.TryRemove(kv.Key, out _);

        // старые страйки самоугасают
        foreach (var kv in _strikes.ToArray())
            if (now - kv.Value.First > _strikeWindow) _strikes.TryRemove(kv.Key, out _);
    }

    private static string Key(ProxyInfo p) => $"{p.Host}:{p.Port}:{p.Username}:{p.Password}";
}
