using System.Net;
using System.Net.Sockets;
using LLM.HHData.Config;
using LLM.HHData.Services;
using Microsoft.Extensions.Options;

namespace LLM.HHData.Http;

public interface IHttpSender
{
    Task<HttpResponseMessage> SendAsync(HttpRequestMessage req, CancellationToken ct);
    Task<string> GetStringAsync(string url, string accept, CancellationToken ct);
}

public sealed class HttpSender : IHttpSender
{
    private readonly ISystemLogService _log;
    private readonly HttpClient _clientSite; // для sitemap
    private readonly HttpClient _clientApi;  // для api.hh.ru (с ротацией прокси)   

    private readonly TimeSpan _minDelay;
    private readonly int _maxRetries;
    private readonly int _retryInitialMs;

    private readonly string? _rawCookieHeader;
    private readonly AppConfig _cfg;
    private readonly IProxyProvider _proxyProvider;

    private readonly ProxyInfo? _fixedProxy;

    public HttpSender(ISystemLogService log, IOptions<AppConfig> cfgOpt, IProxyProvider proxyProvider, ProxyInfo? fixedProxy = null)
    {
        _log = log;
        _cfg = cfgOpt.Value;
        _proxyProvider = proxyProvider;
        _fixedProxy = fixedProxy;

        // Куки только вручную (чтобы не утекали на API)
        var handlerSite = new SocketsHttpHandler
        {
            AutomaticDecompression = DecompressionMethods.GZip | DecompressionMethods.Deflate,
            AllowAutoRedirect = true,
            ConnectTimeout = TimeSpan.FromSeconds(_cfg.Http.TimeoutSeconds),
            UseCookies = false
        };
        _clientSite = new HttpClient(handlerSite) { Timeout = TimeSpan.FromSeconds(_cfg.Http.TimeoutSeconds) };

        // API-клиент: на каждый коннект — новый SOCKS5 из провайдера
        var handlerApi = new SocketsHttpHandler
        {
            AutomaticDecompression = DecompressionMethods.GZip | DecompressionMethods.Deflate,
            AllowAutoRedirect = true,
            ConnectTimeout = TimeSpan.FromSeconds(_cfg.Http.TimeoutSeconds),
            UseCookies = false,
            PooledConnectionLifetime = TimeSpan.FromMinutes(2),
            ConnectCallback = async (ctx, ct) =>
            {
                try
                {
                    if (!_cfg.Http.UseProxy)
                    {
                        var tcp = new TcpClient();
                        await tcp.ConnectAsync(ctx.DnsEndPoint.Host, ctx.DnsEndPoint.Port, ct);
                        return tcp.GetStream();
                    }

                    var p = _fixedProxy ?? _proxyProvider.GetRandomProxy();

                    if (p == null)
                    {
                        var tcp = new TcpClient();
                        await tcp.ConnectAsync(ctx.DnsEndPoint.Host, ctx.DnsEndPoint.Port, ct);
                        return tcp.GetStream();
                    }

                    return await Socks5Connector.ConnectAsync(
                        p.Host, p.Port,
                        ctx.DnsEndPoint.Host, ctx.DnsEndPoint.Port,
                        p.Username, p.Password,
                        ct);
                }
                catch (OperationCanceledException oce) when (ct.IsCancellationRequested)
                {
                    throw new TimeoutException("PROXY_CONNECT_TIMEOUT", oce);
                }
                catch (EndOfStreamException eos)
                {
                    await _log.ExceptionAsync(eos);
                    // прокси рвёт соединение при рукопожатии → баним
                    throw new OperationCanceledException("Proxy banned: SOCKS EOF", eos);
                }
                catch (IOException ioex) when (ioex.Message.Contains("SOCKS5 auth failed", StringComparison.OrdinalIgnoreCase))
                {
                    await _log.ExceptionAsync(ioex);
                    // неверные креды на SOCKS → баним прокси
                    throw new OperationCanceledException("Proxy banned: SOCKS5 auth failed", ioex);
                }
                catch (Exception ex)
                {
                    var addr = _fixedProxy == null ? " (no proxy)" : _fixedProxy.Host + ":" + _fixedProxy.Port;

                    await _log.ErrorAsync($"BEFORE EXCEPTION PROXY ADDR: {addr}");
                    await _log.ExceptionAsync(ex);

                    throw new IOException("PROXY_CONNECT_FAILED", ex);
                }
            }
        };

        _clientApi = new HttpClient(handlerApi)
        {
            Timeout = TimeSpan.FromSeconds(_cfg.Http.TimeoutSeconds),
            DefaultRequestVersion = HttpVersion.Version11,
            DefaultVersionPolicy = HttpVersionPolicy.RequestVersionExact
        };

        // общие заголовки
        foreach (var c in new[] { _clientSite, _clientApi })
        {
            c.DefaultRequestHeaders.UserAgent.ParseAdd(_cfg.Http.UserAgent);
            c.DefaultRequestHeaders.Accept.ParseAdd("application/json, application/xml;q=0.9");
            c.DefaultRequestHeaders.AcceptEncoding.ParseAdd("gzip, deflate");
        }

        // rps → _minDelay (пер-прокси или глобальный)
        double rps = 0;
        if (_fixedProxy != null)
            rps = _cfg.Http.PerProxyRps > 0 ? _cfg.Http.PerProxyRps : _cfg.Http.RequestsPerSecond; // legacy fallback
        else
            rps = _cfg.Http.GlobalRps > 0 ? _cfg.Http.GlobalRps : _cfg.Http.RequestsPerSecond;

        _minDelay = rps > 0 ? TimeSpan.FromSeconds(1.0 / rps) : TimeSpan.Zero;
        _maxRetries = _cfg.Http.MaxRetries;
        _retryInitialMs = Math.Max(50, _cfg.Http.RetryInitialDelayMs);

        if (_cfg.Cookies.Enabled) _rawCookieHeader = CookieLoader.BuildRawHeader(_cfg.Cookies.FilePath);
    }

    // тактовка слотов старта запросов
    private readonly SemaphoreSlim _gate = new(1, 1);
    private DateTimeOffset _nextAt = DateTimeOffset.MinValue;

    public async Task<HttpResponseMessage> SendAsync(HttpRequestMessage req, CancellationToken ct)
    {
        var isApi = string.Equals(req.RequestUri!.Host, "api.hh.ru", StringComparison.OrdinalIgnoreCase);
        var client = isApi ? _clientApi : _clientSite;

        // ---- ТРОТТЛИНГ: резервируем слот ДО выхода из gate
        TimeSpan wait;
        await _gate.WaitAsync(ct).ConfigureAwait(false);
        try
        {
            var now = DateTimeOffset.UtcNow;
            wait = (now < _nextAt) ? (_nextAt - now) : TimeSpan.Zero;

            var scheduledStart = (wait > TimeSpan.Zero) ? _nextAt : now;
            _nextAt = scheduledStart + _minDelay;
        }
        finally
        {
            _gate.Release();
        }

        if (wait > TimeSpan.Zero)
            await Task.Delay(wait, ct).ConfigureAwait(false);

        var attempt = 0;
        var delay = TimeSpan.FromMilliseconds(_retryInitialMs);

        while (true)
        {
            using var attemptReq = CloneRequest(req); // клон на каждую попытку

            // селективные куки (для sitemap)
            attemptReq.Headers.Remove("Cookie");
            if (!isApi && _rawCookieHeader != null)
                attemptReq.Headers.Add("Cookie", _rawCookieHeader);

            // для API — принудительно закрывать соединение, и Bearer на attemptReq
            if (isApi)
            {
                attemptReq.Version = HttpVersion.Version11;
                attemptReq.VersionPolicy = HttpVersionPolicy.RequestVersionExact;
                attemptReq.Headers.ConnectionClose = true;

                if (!string.IsNullOrWhiteSpace(_cfg.Http.BearerToken) && attemptReq.Headers.Authorization is null)
                {
                    attemptReq.Headers.Authorization =
                        new System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", _cfg.Http.BearerToken);
                }
            }

            HttpResponseMessage resp;

            try
            {
                resp = await client.SendAsync(attemptReq, HttpCompletionOption.ResponseHeadersRead, ct)
                                   .ConfigureAwait(false);
            }
            catch when (attempt < _maxRetries)
            {
                attempt++;
                await Task.Delay(Backoff(delay, attempt), ct).ConfigureAwait(false);
                continue;
            }

            if (IsOkOrRedirect(resp.StatusCode)) return resp;

            if (resp.StatusCode == HttpStatusCode.Forbidden) // 403
            {
                var code = (int)resp.StatusCode; var reason = resp.ReasonPhrase;
                resp.Dispose();
                // важный сигнал для верхнего уровня — это «плохой» прокси
                throw new OperationCanceledException("Proxy banned: HTTP 403 Forbidden");
            }

            if (IsFatal(resp.StatusCode))
            {
                var code = (int)resp.StatusCode; var reason = resp.ReasonPhrase;
                resp.Dispose();
                throw new HttpRequestException($"Fatal HTTP {code} {reason}");
            }

            if (IsRetryable(resp.StatusCode) && attempt < _maxRetries)
            {
                var ra = resp.Headers.RetryAfter?.Delta ?? Backoff(delay, ++attempt);
                resp.Dispose();
                await Task.Delay(ra, ct).ConfigureAwait(false);
                continue;
            }

            var ccode = (int)resp.StatusCode; var rs = resp.ReasonPhrase;
            resp.Dispose();
            throw new HttpRequestException($"HTTP {ccode} {rs}, retries exhausted");
        }
    }

    static bool IsOkOrRedirect(HttpStatusCode c) =>
        (int)c is >= 200 and <= 299 ||
        c is HttpStatusCode.MovedPermanently or HttpStatusCode.Redirect or HttpStatusCode.TemporaryRedirect or HttpStatusCode.PermanentRedirect;

    static bool IsFatal(HttpStatusCode c) =>
        c is HttpStatusCode.NotFound or HttpStatusCode.Gone; // 403 вынесли выше


    static bool IsRetryable(HttpStatusCode c) =>
        c is (HttpStatusCode)429 or HttpStatusCode.InternalServerError or HttpStatusCode.BadGateway
         or HttpStatusCode.ServiceUnavailable or HttpStatusCode.GatewayTimeout;

    static TimeSpan Backoff(TimeSpan baseDelay, int attempt) =>
        TimeSpan.FromMilliseconds(baseDelay.TotalMilliseconds * Math.Pow(2, attempt - 1) + Random.Shared.Next(100, 400));

    // аккуратный клон (с контентом, если есть)
    private static HttpRequestMessage CloneRequest(HttpRequestMessage req)
    {
        var clone = new HttpRequestMessage(req.Method, req.RequestUri);

        foreach (var h in req.Headers)
            clone.Headers.TryAddWithoutValidation(h.Key, h.Value);

        if (req.Content != null)
        {
            var ms = new MemoryStream();
            req.Content.CopyToAsync(ms).GetAwaiter().GetResult();
            ms.Position = 0;
            var contentClone = new StreamContent(ms);

            foreach (var h in req.Content.Headers)
                contentClone.Headers.TryAddWithoutValidation(h.Key, h.Value);

            clone.Content = contentClone;
        }

        foreach (var opt in req.Options)
            clone.Options.Set(new HttpRequestOptionsKey<object?>(opt.Key), opt.Value);

        return clone;
    }

    public async Task<string> GetStringAsync(string url, string accept, CancellationToken ct)
    {
        using var req = new HttpRequestMessage(HttpMethod.Get, url);
        req.Headers.Accept.Clear();
        req.Headers.Accept.ParseAdd(accept);
        using var resp = await SendAsync(req, ct);

        if ((int)resp.StatusCode == 407 || resp.StatusCode == HttpStatusCode.Forbidden /*403*/)
            throw new OperationCanceledException("Proxy banned: PROXY_AUTH_OR_FORBIDDEN");

        if (resp.StatusCode == (HttpStatusCode)429 || (int)resp.StatusCode >= 500)
            throw new IOException("REMOTE_RATE_OR_5XX");

        return await resp.Content.ReadAsStringAsync(ct);
    }

    public int PreviewCookieFileCount() => CookieLoader.CountCookies(_cfg.Cookies.FilePath);
}
