using LLM.HHData.Services;
using Npgsql;

namespace LLM.HHData.Db;

public interface IRawStore
{
    Task UpsertEmployerAsync(long employerId, string rawJson, CancellationToken ct);
    Task UpsertVacancyAsync(long vacancyId, long employerId, string rawJson, CancellationToken ct);
    Task<HashSet<long>> LoadKnownEmployerIdsAsync(CancellationToken ct);
    Task<HashSet<long>> LoadKnownVacancyIdsAsync(CancellationToken ct);
    Task<Dictionary<long, DateTimeOffset>> LoadVacancyLastSeenAsync(CancellationToken ct);
    Task TouchVacancyAsync(long vacancyId, CancellationToken ct);
}


public sealed class BatchedRawStoreAdapter(IBatchedRawStore inner, IDbFactory db, ISystemLogService log) : IRawStore
{
    private readonly IBatchedRawStore _inner = inner;
    private readonly IDbFactory _db = db;
    private readonly ISystemLogService _log = log;

    public Task UpsertVacancyAsync(long vacancyId, long employerId, string json, CancellationToken ct)
        => _inner.UpsertVacancyAsync(vacancyId, employerId, json, ct);

    public Task UpsertEmployerAsync(long employerId, string json, CancellationToken ct)
        => _inner.UpsertEmployerAsync(employerId, json, ct);

    public Task TouchVacancyAsync(long vacancyId, CancellationToken ct)
        => _inner.TouchVacancyAsync(vacancyId, ct);

    public Task TouchEmployerAsync(long employerId, CancellationToken ct)
        => _inner.TouchEmployerAsync(employerId, ct);

    public async Task<HashSet<long>> LoadKnownEmployerIdsAsync(CancellationToken ct)
    {
        try
        {
            var set = new HashSet<long>();
            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var cmd = conn.CreateCommand();
            cmd.CommandText = $@"select employer_id from {_db.Schema}.employers_raw;";
            await using var rdr = await cmd.ExecuteReaderAsync(ct);
            while (await rdr.ReadAsync(ct)) set.Add(rdr.GetInt64(0));
            return set;
        }
        catch (PostgresException ex)
        {
            await _log.ExceptionAsync(ex);

            throw;
        }
    }

    public async Task<HashSet<long>> LoadKnownVacancyIdsAsync(CancellationToken ct)
    {
        try
        {


            var set = new HashSet<long>();
            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var cmd = conn.CreateCommand();
            cmd.CommandText = $@"select vacancy_id from {_db.Schema}.vacancies_raw;";
            await using var rdr = await cmd.ExecuteReaderAsync(ct);
            while (await rdr.ReadAsync(ct)) set.Add(rdr.GetInt64(0));
            return set;
        }
        catch (PostgresException ex)
        {
            await _log.ExceptionAsync(ex);

            throw;
        }
    }

    public async Task<Dictionary<long, DateTimeOffset>> LoadVacancyLastSeenAsync(CancellationToken ct)
    {
        try
        {

            var map = new Dictionary<long, DateTimeOffset>();
            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var cmd = conn.CreateCommand();
            cmd.CommandText = $@"select vacancy_id, last_seen_at from {_db.Schema}.vacancies_raw;";
            await using var rdr = await cmd.ExecuteReaderAsync(ct);
            while (await rdr.ReadAsync(ct))
            {
                var id = rdr.GetInt64(0);
                var ts = rdr.GetFieldValue<DateTimeOffset>(1);
                map[id] = ts;
            }
            return map;
        }
        catch (PostgresException ex)
        {
            await _log.ExceptionAsync(ex);

            throw;
        }
    }
}