using System;
using System.Threading;
using System.Threading.Tasks;
using LLM.HHData.Config;
using Npgsql;

namespace LLM.HHData.Db
{
    public interface IDbFactory : IAsyncDisposable, IDisposable
    {
        string Schema { get; }
        NpgsqlConnection OpenConnection();
        ValueTask<NpgsqlConnection> OpenConnectionAsync(CancellationToken ct = default);
        NpgsqlDataSource DataSource { get; }
    }

    public sealed class DbFactory : IDbFactory
    {
        private readonly NpgsqlDataSource _ds;
        public string Schema { get; }

        // параметры пула вынесены в константу
        private const string PoolingParams =
            "Pooling=true;" +
            "Minimum Pool Size=20;" +
            "Maximum Pool Size=400;" +
            "Connection Idle Lifetime=300;" +
            "Connection Pruning Interval=30;" +
            "Timeout=15;" +
            "Command Timeout=60;" +
            "Max Auto Prepare=100;" +
            "Auto Prepare Min Usages=2;" +
            "Multiplexing=true";

        public DbFactory(AppConfig cfg)
        {
            // добавляем параметры пула к строке подключения
            var cs = cfg.Database.ConnectionString;
            if (!cs.Contains("Pooling", StringComparison.OrdinalIgnoreCase))
                cs = cs.TrimEnd(';') + ";" + PoolingParams;

            var builder = new NpgsqlDataSourceBuilder(cs);

            _ds = builder.Build();
            Schema = cfg.Database.Schema;
        }

        public NpgsqlDataSource DataSource => _ds;

        public NpgsqlConnection OpenConnection() => _ds.OpenConnection();

        public ValueTask<NpgsqlConnection> OpenConnectionAsync(CancellationToken ct = default)
            => _ds.OpenConnectionAsync(ct);

        public void Dispose() => _ds.Dispose();

        public async ValueTask DisposeAsync()
        {
            await _ds.DisposeAsync();
        }
    }
}
