using System.Collections.Concurrent;
using System.Threading.Channels;
using LLM.HHData.Config;
using LLM.HHData.Services;
using Microsoft.Extensions.Options;
using Npgsql;

using static LLM.HHData.Config.AppConfig;

namespace LLM.HHData.Db;

public interface IBatchedRawStore : IAsyncDisposable
{
    // те же методы, что и в IRawStore, но они только enqueue
    Task UpsertVacancyAsync(long vacancyId, long employerId, string json, CancellationToken ct);
    Task UpsertEmployerAsync(long employerId, string json, CancellationToken ct);
    Task TouchVacancyAsync(long vacancyId, CancellationToken ct);
    Task TouchEmployerAsync(long employerId, CancellationToken ct);
}

public sealed class BatchedRawStore : IBatchedRawStore
{
    private readonly IDbFactory _db;
    private readonly DatabaseConfig _dbcfg;
    private readonly ISystemLogService _log;

    private readonly Channel<VacItem> _vacChan;
    private readonly Channel<EmpItem> _empChan;
    private readonly Channel<long> _touchVacChan;
    private readonly Channel<long> _touchEmpChan;

    private readonly Task _vacFlushTask;
    private readonly Task _empFlushTask;
    private readonly Task _touchFlushTask;

    private readonly TimeSpan _flushInterval;
    private readonly int _vacMax;
    private readonly int _empMax;
    private readonly int _touchMax;
    private readonly TimeSpan _touchDebounce;

    private readonly ConcurrentDictionary<long, DateTimeOffset> _touchVacSeen = new();
    private readonly ConcurrentDictionary<long, DateTimeOffset> _touchEmpSeen = new();

    private sealed record VacItem(long VacancyId, long EmployerId, string Json);
    private sealed record EmpItem(long EmployerId, string Json);

    public BatchedRawStore(IDbFactory db, IOptions<AppConfig> cfgOpt, ISystemLogService log)
    {
        _db = db;
        _dbcfg = cfgOpt.Value.Database;
        _log = log;

        _vacMax = _dbcfg.Batch.VacancyMaxBatch;
        _empMax = _dbcfg.Batch.EmployerMaxBatch;
        _touchMax = _dbcfg.Batch.TouchMaxBatch;
        _flushInterval = TimeSpan.FromMilliseconds(_dbcfg.Batch.FlushIntervalMs);
        _touchDebounce = TimeSpan.FromMinutes(_dbcfg.Batch.TouchDebounceMinutes);

        var cap = _dbcfg.Batch.MaxPendingItems;

        _vacChan = Channel.CreateBounded<VacItem>(new BoundedChannelOptions(cap) { FullMode = BoundedChannelFullMode.Wait });
        _empChan = Channel.CreateBounded<EmpItem>(new BoundedChannelOptions(cap) { FullMode = BoundedChannelFullMode.Wait });
        _touchVacChan = Channel.CreateBounded<long>(new BoundedChannelOptions(cap) { FullMode = BoundedChannelFullMode.DropOldest });
        _touchEmpChan = Channel.CreateBounded<long>(new BoundedChannelOptions(cap) { FullMode = BoundedChannelFullMode.DropOldest });

        _vacFlushTask = Task.Run(() => FlushVacLoopAsync(CancellationToken.None));
        _empFlushTask = Task.Run(() => FlushEmpLoopAsync(CancellationToken.None));
        _touchFlushTask = Task.Run(() => FlushTouchLoopAsync(CancellationToken.None));
    }

    public async Task UpsertVacancyAsync(long vacancyId, long employerId, string json, CancellationToken ct)
        => await _vacChan.Writer.WriteAsync(new VacItem(vacancyId, employerId, json), ct);

    public async Task UpsertEmployerAsync(long employerId, string json, CancellationToken ct)
        => await _empChan.Writer.WriteAsync(new EmpItem(employerId, json), ct);

    public async Task TouchVacancyAsync(long vacancyId, CancellationToken ct)
    {
        var now = DateTimeOffset.UtcNow;
        var last = _touchVacSeen.GetOrAdd(vacancyId, now - _touchDebounce - TimeSpan.FromSeconds(1));
        if (now - last < _touchDebounce) return;
        _touchVacSeen[vacancyId] = now;
        await _touchVacChan.Writer.WriteAsync(vacancyId, ct);
    }

    public async Task TouchEmployerAsync(long employerId, CancellationToken ct)
    {
        var now = DateTimeOffset.UtcNow;
        var last = _touchEmpSeen.GetOrAdd(employerId, now - _touchDebounce - TimeSpan.FromSeconds(1));
        if (now - last < _touchDebounce) return;
        _touchEmpSeen[employerId] = now;
        await _touchEmpChan.Writer.WriteAsync(employerId, ct);
    }

    // ─────────────────────────────────────────────────────────────

    private async Task FlushVacLoopAsync(CancellationToken ct)
    {
        var buffer = new List<VacItem>(_vacMax);
        var nextFlush = DateTime.UtcNow + _flushInterval;

        while (await _vacChan.Reader.WaitToReadAsync(ct))
        {
            while (_vacChan.Reader.TryRead(out var item))
            {
                buffer.Add(item);
                if (buffer.Count >= _vacMax || DateTime.UtcNow >= nextFlush)
                {
                    await FlushVacanciesAsync(buffer, ct);
                    buffer.Clear();
                    nextFlush = DateTime.UtcNow + _flushInterval;
                }
            }
        }

        if (buffer.Count > 0)
            await FlushVacanciesAsync(buffer, ct);
    }

    private async Task FlushEmpLoopAsync(CancellationToken ct)
    {
        var buffer = new List<EmpItem>(_empMax);
        var nextFlush = DateTime.UtcNow + _flushInterval;

        while (await _empChan.Reader.WaitToReadAsync(ct))
        {
            while (_empChan.Reader.TryRead(out var item))
            {
                buffer.Add(item);
                if (buffer.Count >= _empMax || DateTime.UtcNow >= nextFlush)
                {
                    await FlushEmployersAsync(buffer, ct);
                    buffer.Clear();
                    nextFlush = DateTime.UtcNow + _flushInterval;
                }
            }
        }

        if (buffer.Count > 0)
            await FlushEmployersAsync(buffer, ct);
    }

    private async Task FlushTouchLoopAsync(CancellationToken ct)
    {
        var vac = new List<long>(_touchMax);
        var emp = new List<long>(_touchMax);
        var nextFlush = DateTime.UtcNow + _flushInterval;

        // мультиплексим оба канала разом: простой пуллер
        while (true)
        {
            var readAny = false;

            while (_touchVacChan.Reader.TryRead(out var v))
            {
                vac.Add(v);
                readAny = true;
                if (vac.Count >= _touchMax) break;
            }
            while (_touchEmpChan.Reader.TryRead(out var e))
            {
                emp.Add(e);
                readAny = true;
                if (emp.Count >= _touchMax) break;
            }

            var timeHit = DateTime.UtcNow >= nextFlush;
            if (readAny && (vac.Count >= _touchMax || emp.Count >= _touchMax || timeHit))
            {
                await FlushTouchesAsync(vac, emp, ct);
                vac.Clear();
                emp.Clear();
                nextFlush = DateTime.UtcNow + _flushInterval;
            }

            // если ничего не прочитали — чуть подождём
            if (!readAny)
                await Task.Delay(50, ct);
        }
    }

    // ─────────────────────────────────────────────────────────────
    // Фактический флаш в БД

    private async Task FlushVacanciesAsync(List<VacItem> batch, CancellationToken ct)
    {
        if (batch.Count == 0) return;
        try
        {
            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var tx = await conn.BeginTransactionAsync(ct);

            // вариант 1: NpgsqlBatch из upsert-ов
            var nb = new NpgsqlBatch(conn)
            {
                Transaction = tx
            };

            string sql = $@"
            INSERT INTO {_dbcfg.Schema}.vacancies_raw (vacancy_id, employer_id, raw_json, last_seen_at)
            VALUES (@v, @e, @p, now())
            ON CONFLICT (vacancy_id) DO UPDATE
            SET employer_id = EXCLUDED.employer_id,
                raw_json    = EXCLUDED.raw_json,
                last_seen_at= now();";

            foreach (var it in batch)
            {
                var cmd = new NpgsqlBatchCommand(sql);
                cmd.Parameters.AddWithValue("v", it.VacancyId);
                cmd.Parameters.AddWithValue("e", it.EmployerId);
                cmd.Parameters.AddWithValue("p", NpgsqlTypes.NpgsqlDbType.Jsonb, it.Json);
                nb.BatchCommands.Add(cmd);
            }


            await nb.ExecuteNonQueryAsync(ct);
            await tx.CommitAsync(ct);
        }
        catch (PostgresException ex)
        {
            await _log.ErrorAsync(
            $"[DB] Batch FAIL sqlstate={ex.SqlState} code={ex.SqlState} " +
            $"msg='{ex.MessageText}' detail='{ex.Detail}' where='{ex.Where}' " +
            $"hint='{ex.Hint}' schema='{ex.SchemaName}' table='{ex.TableName}' " +
            $"batch={batch.Count}");

            throw;
        }
    }

    private async Task FlushEmployersAsync(List<EmpItem> batch, CancellationToken ct)
    {
        if (batch.Count == 0) return;

        try
        {
            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var tx = await conn.BeginTransactionAsync(ct);

            var nb = new NpgsqlBatch(conn) { Transaction = tx };

            string sql = $@"
            INSERT INTO {_dbcfg.Schema}.employers_raw (employer_id, raw_json, last_seen_at)
            VALUES (@e, @p, now())
            ON CONFLICT (employer_id) DO UPDATE
            SET raw_json    = EXCLUDED.raw_json,
                last_seen_at= now();";

            foreach (var it in batch)
            {
                var cmd = new NpgsqlBatchCommand(sql);
                cmd.Parameters.AddWithValue("e", it.EmployerId);
                cmd.Parameters.AddWithValue("p", NpgsqlTypes.NpgsqlDbType.Jsonb, it.Json);
                nb.BatchCommands.Add(cmd);
            }

            await nb.ExecuteNonQueryAsync(ct);
            await tx.CommitAsync(ct);
        }
        catch (PostgresException ex)
        {
            await _log.ErrorAsync(
            $"[DB] Batch FAIL sqlstate={ex.SqlState} code={ex.SqlState} " +
            $"msg='{ex.MessageText}' detail='{ex.Detail}' where='{ex.Where}' " +
            $"hint='{ex.Hint}' schema='{ex.SchemaName}' table='{ex.TableName}' " +
            $"batch={batch.Count}");

            throw;
        }
    }

    private async Task FlushTouchesAsync(List<long> vacIds, List<long> empIds, CancellationToken ct)
    {
        if (vacIds.Count == 0 && empIds.Count == 0) return;
        try
        {

            await using var conn = await _db.OpenConnectionAsync(ct);
            await using var tx = await conn.BeginTransactionAsync(ct);
            var nb = new NpgsqlBatch(conn) { Transaction = tx };

            if (vacIds.Count > 0)
            {
                string sql = $@"
                    UPDATE {_dbcfg.Schema}.vacancies_raw AS t
                    SET last_seen_at = now()
                    FROM (SELECT UNNEST(@ids) AS id) s
                    WHERE t.vacancy_id = s.id;";
                var cmd = new NpgsqlBatchCommand(sql);
                cmd.Parameters.AddWithValue("ids", vacIds);
                nb.BatchCommands.Add(cmd);
            }

            if (empIds.Count > 0)
            {
                string sql = $@"
                    UPDATE {_dbcfg.Schema}.employers_raw AS t
                    SET last_seen_at = now()
                    FROM (SELECT UNNEST(@ids) AS id) s
                    WHERE t.employer_id = s.id;";
                var cmd = new NpgsqlBatchCommand(sql);
                cmd.Parameters.AddWithValue("ids", empIds);
                nb.BatchCommands.Add(cmd);
            }

            await nb.ExecuteNonQueryAsync(ct);
            await tx.CommitAsync(ct);
        }
        catch (PostgresException ex)
        {
            await _log.ErrorAsync(
            $"[DB] Batch FAIL sqlstate={ex.SqlState} code={ex.SqlState} " +
            $"msg='{ex.MessageText}' detail='{ex.Detail}' where='{ex.Where}' " +
            $"hint='{ex.Hint}' schema='{ex.SchemaName}' table='{ex.TableName}'");

            throw;
        }
    }

    // ─────────────────────────────────────────────────────────────

    public async ValueTask DisposeAsync()
    {
        _vacChan.Writer.TryComplete();
        _empChan.Writer.TryComplete();
        _touchVacChan.Writer.TryComplete();
        _touchEmpChan.Writer.TryComplete();

        // даём флашерам завершить очереди
        var tasks = new[] { _vacFlushTask, _empFlushTask, _touchFlushTask };
        await Task.WhenAny(Task.WhenAll(tasks), Task.Delay(2000));
    }
}