﻿using AiLogia.Bot.Core;
using AiLogia.Bot.Data;
using AiLogia.Bot.Diagnostics;
using AiLogia.Bot.Handlers;
using AiLogia.Bot.Services;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Logging;
using Telegram.Bot;

internal class Program
{
    private static async Task Main(string[] args)
    {
        Console.Clear();

        var builder = Host.CreateDefaultBuilder(args)
       .ConfigureAppConfiguration(cfg =>
       {
           cfg.AddJsonFile("config.json", optional: false, reloadOnChange: true);
       })
       .ConfigureLogging(logging =>
       {
           logging.ClearProviders();
           logging.AddConsole();
           logging.SetMinimumLevel(LogLevel.Warning);
       })
       .ConfigureServices((context, services) =>
       {
           var config = context.Configuration;
           var botToken = config["BotToken"];

           // client
           services.AddSingleton<ITelegramBotClient>(new TelegramBotClient(botToken));

           // logs, stats
           services.AddSingleton<ISystemLogService, SystemLogService>();


           services.AddSingleton<IUserActionTracker, UserActionTracker>();

           // data
           services.AddSingleton<ITestsRepository, TestsRepository>();
           services.AddSingleton<IRecommendationsRepository, RecommendationsRepository>();

           // commands
           services.AddSingleton<ICommandsRouter, CommandsRouter>();
           services.AddSingleton<ICommandHandler, StartCommandHandler>();

           services.AddSingleton<ICommandsRouter, CommandsRouter>();
           services.AddSingleton<ICommandHandler, StartCommandHandler>();
           services.AddSingleton<ICommandHandler, TestsCommandHandler>();
           services.AddSingleton<ICommandHandler, ShareCommandHandler>();
           services.AddSingleton<ICommandHandler, MeCommandHandler>();

           services.AddSingleton<ICommandHandler, RecomendationsCommandHandler>();

           // fb
           services.AddSingleton<IFeedbackStateService, InMemoryFeedbackStateService>();
           services.AddSingleton<ICommandHandler, FeedbackCommandHandler>();
           services.AddSingleton<ICommandHandler, CancelFeedbackCommandHandler>();
           services.AddSingleton<IMessageHandler, FeedbackMessageHandler>();

           // handlers
           services.AddSingleton<ICallbackHandler, TestSelectedHandler>();
           services.AddSingleton<ICallbackHandler, QuestionFlowHandler>();



           // auth

           services.AddSingleton<IAuthClient>(sp =>
            {
                var cfg = sp.GetRequiredService<IConfiguration>();
                var http = new HttpClient { BaseAddress = new Uri(cfg["AuthService:BaseUrl"]!) };
                return new AuthClient(http, cfg);
            });

           services.AddSingleton<ICommandHandler, AuthCommandHandler>();

           // main app host
           services.AddSingleton<TelegramBotHost>();
           services.AddHostedService<TelegramBotHost>();
       });

        await builder.Build().RunAsync();
    }
}