using System.Text;
using AiLogia.Bot.Data;
using AiLogia.Bot.Diagnostics;
using AiLogia.Bot.Services;
using Microsoft.Extensions.Configuration;
using Telegram.Bot;
using Telegram.Bot.Types;
using Telegram.Bot.Types.Enums;

namespace AiLogia.Bot.Handlers;


public class TestsCommandHandler(
    IConfiguration configuration,
    ITelegramBotClient bot,
    ISystemLogService log,
    IUserActionTracker tracker,
    ITestsRepository repo ) : ICommandHandler
{
    public string Command => "/tests";

    private readonly ITelegramBotClient _bot = bot;
    private readonly ISystemLogService _log = log;
    private readonly IUserActionTracker _tracker = tracker;
    private readonly ITestsRepository _repo = repo;

    private readonly string _botUsername = configuration.GetValue<string>("BotTgName") ?? throw new InvalidOperationException("BotTgName is not set");

    public async Task HandleAsync(Message message)
    {
        await _tracker.TrackAsync("command:tests", message);

        var userId = message.From?.Id ?? 0;

        if (userId == 0)
        {
            await _bot.SendMessage(
               chatId: message.Chat.Id,
               text: "😢 Пока нет доступных тестов."
           );
            return;
        }

        var userAccess = await _repo.GetOrCreateUserAccessAsync(userId);
        var tests = await _repo.GetAvailableTestsAsync(userAccess);

        if (tests.Count == 0)
        {
            await _bot.SendMessage(
                chatId: message.Chat.Id,
                text: "😢 Пока нет доступных тестов."
            );
            return;
        }

        var grouped = tests
            .GroupBy(t => t.GroupTitle ?? "Без категории")
            .OrderBy(g => g.Key)
            .ToList();

        // custom messages


        var sb = new StringBuilder();
        sb.AppendLine("👋 Классно, что вы решили пройти тест!");
        sb.AppendLine("📋 Вот что у нас есть на сегодня — выбирайте, какой пройти:\n");

        foreach (var group in grouped)
        {
            sb.AppendLine($"🔸 *{group.Key}*");
            foreach (var test in group)
            {
                var escapedTitle = test.Title;
                var link = $"https://t.me/{_botUsername}?start=test_{test.Id}";
                sb.AppendLine($"  ➜ [{escapedTitle}]({link})");
            }
            sb.AppendLine();
        }

        sb.AppendLine("🧘‍♀️ Мы заботимся о вашем настроении и саморазвитии, и постоянно добавляем новые тесты! 👀 Заглядывайте регулярно, Вам точно понравится 💙");

        await _bot.SendMessage(
            chatId: message.Chat.Id,
            text: sb.ToString(),
            parseMode: ParseMode.Markdown
        );
    }
}