using System.Text;
using AiLogia.Bot.Data;
using AiLogia.Bot.Diagnostics;
using AiLogia.Bot.Services;
using AiLogia.Bot.Utils;
using Microsoft.Extensions.Configuration;
using Telegram.Bot;
using Telegram.Bot.Types;
using Telegram.Bot.Types.Enums;

namespace AiLogia.Bot.Handlers;

public class MeCommandHandler(
    IConfiguration configuration, ITelegramBotClient bot, ISystemLogService log,
    IUserActionTracker tracker, ITestsRepository repo) : ICommandHandler
{
    public string Command => "/me";
    private readonly ITelegramBotClient _bot = bot;
    private readonly ISystemLogService _log = log;
    private readonly IUserActionTracker _tracker = tracker;
    private readonly ITestsRepository _repo = repo;
    private readonly string _botUsername = configuration.GetValue<string>("BotTgName") ?? throw new InvalidOperationException("BotTgName is not set");
    const bool debug = false;

    public async Task HandleAsync(Message message)
    {
        await _tracker.TrackAsync("command:me", message);

        var userId = message.From?.Id ?? 0;
        var username = message.From?.Username ?? default!;
        var results = await _repo.GetUserResultsAsync(userId);

        if (results.Count == 0)
        {
            var text = "🙁 Вы пока не прошли ни одного теста.\n" +
                       "👉 Используйте команду /tests чтобы выбрать какой тест пройти.";

            await _bot.SendMessage(chatId: message.Chat.Id, text: text, parseMode: ParseMode.Markdown);

            return;
        }

        var sb = new StringBuilder();
        sb.AppendLine("📊 *Ваши результаты:*");
        sb.AppendLine();

        foreach (var res in results.OrderByDescending(r => r.CompletedAt))
        {
            res.BotUsername = _botUsername;

            var resultsTitle = res.TestTitle.Replace("\\n", Environment.NewLine);
            var resultsText = res.ResultText.Replace("\\n", Environment.NewLine);

            if (res.TestAccessMode == "public")
            {
                // обычный тест - просто форимируем результат как обычно

                sb.AppendLine($"📅 {res.CompletedAt:dd.MM.yyyy HH:mm}");
                sb.AppendLine($"🧪 {resultsTitle} — *{resultsText}*");

                if (!string.IsNullOrWhiteSpace(res.Interpretation))
                {
                    var interpretation = res.Interpretation.Replace("\\n", Environment.NewLine);

                    sb.AppendLine($"📝 {interpretation}");
                }

                sb.AppendLine();

            }
            else if (res.TestAccessMode == "channel_subscription_required")
            {

                // тест за подписку, проверим есть ли пользак в подписчиках, если нет - скроем результат и попросим подписаться
                
            }
        }

        var chunks = TelegramMessageHelper.SplitMessageIntoChunks(sb.ToString());

        foreach (var chunk in chunks)
        {
            await _bot.SendMessage(
                chatId: message.Chat.Id,
                text: chunk,
                parseMode: ParseMode.Markdown
            );
        }      
    }
}