using AiLogia.Bot.Services;
using Microsoft.Extensions.Hosting;
using Telegram.Bot;
using Telegram.Bot.Polling;
using Telegram.Bot.Types;
using Telegram.Bot.Types.Enums;

namespace AiLogia.Bot.Core;

public class TelegramBotHost(ISystemLogService log, ITelegramBotClient bot, ICommandsRouter router) : BackgroundService
{
    private readonly ISystemLogService _log = log;
    private readonly ITelegramBotClient _bot = bot;
    private readonly ICommandsRouter _router = router;

    public ITelegramBotClient Client
    {
        get { return _bot; }
    }

    protected override async Task ExecuteAsync(CancellationToken stoppingToken)
    {
        var receiverOptions = new ReceiverOptions
        {
            AllowedUpdates = Array.Empty<UpdateType>()
        };

        _bot.StartReceiving(
            HandleUpdateAsync,
            HandleErrorAsync,
            receiverOptions,
            cancellationToken: stoppingToken
        );

        await _log.InfoAsync("Bot started");
    }

    private async Task HandleUpdateAsync(ITelegramBotClient bot, Update update, CancellationToken token)
    {
        try
        {
            // 1) Любые входящие сообщения (текст, фото, документ, голос и т.д.)
            if (update.Message is { } message)
            {
                if (message.Text?.StartsWith("/") == true)
                {
                    await _router.RouteCommandAsync(message);
                }
                else
                {
                    // сюда попадут и медиа без подписи, и медиа с подписью
                    await _router.RouteMessageAsync(message);
                }
                return;
            }

            // 2) Колбэки
            if (update.CallbackQuery is { } cb)
            {
                await _router.RouteCallbackAsync(cb);
                return;
            }

            // (опционально: можно логнуть прочие типы апдейтов)
            // _logger.LogDebug("Unhandled update type: {Type}", update.Type);
        }
        catch (Exception ex)
        {
            await HandleErrorAsync(bot, ex, token);
        }
    }

    private async Task HandleErrorAsync(ITelegramBotClient _, Exception ex, CancellationToken token)
    {
        await _log.ExceptionAsync(ex);
    }
}
